﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include "../Common/testLbl_Helper.h"
#include "../Common/testLbl_Util-hardware.nx.h"
#include "../Common/testLbl_Config.h"

#include <nn/os/os_Mutex.h>

/**
 * @brief   現在の輝度設定値、適用された輝度設定値の取得が正しくできるか（共通）
 */
TEST(StateTransition, BrightnessSetting)
{
    nnt::lbl::LblInitializer initializer;

    float brightnessReference = 0.5f;   // 中クラスの輝度設定値
    float brightnessOverwrite = brightnessReference + 0.1f;
    float margin = 0.05f;

    // 取得される現在の輝度設定値は自動輝度調整on/off に関わらない。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    EXPECT_EQ(brightnessReference, lbldfc::GetCurrentBrightnessSetting());
    lbldfc::EnableAutoBrightnessControl();
    EXPECT_EQ(brightnessReference, lbldfc::GetCurrentBrightnessSetting());
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    EXPECT_EQ(brightnessReference, lbldfc::GetCurrentBrightnessSetting());

    // バックライト出力適用後(自動輝度調整off)
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    lbldfc::SetCurrentBrightnessSetting(brightnessOverwrite);
    EXPECT_EQ(brightnessOverwrite, lbldfc::GetCurrentBrightnessSetting());
    // 現在の手動輝度調整時は輝度設定値 != 輝度レベルであるが、
    // 近い値とはなるので、マージンを持たせて緩めにテストする。
    EXPECT_LT(brightnessReference - margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
    EXPECT_GT(brightnessReference + margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
}

/**
 * @brief   現在の輝度設定値、適用された輝度設定値の取得が正しくできるか（要照度センサ）
 */
TEST(StateTransition, BrightnessSetting_RequireAls)
{
    nnt::lbl::LblInitializer initializer;

    ASSERT_TRUE(lbldfc::IsAmbientLightSensorAvailable());

    float brightnessReference = 0.5f;   // 中クラスの輝度設定値
    float sensorReference = 400.0f;     // 中クラスの照度センサーの値
    float brightnessOverwrite = brightnessReference + 0.1f;

    // 自動輝度調整on安定時に適用される輝度設定値。
    lbldfc::EnableAutoBrightnessControl();
    float autoAppliedBrightness = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);

    ASSERT_LT(brightnessReference, brightnessOverwrite);
    ASSERT_NE(brightnessReference, autoAppliedBrightness);
    ASSERT_NE(brightnessOverwrite, autoAppliedBrightness);

    // バックライト出力適用後(自動輝度調整on)
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    lbldfc::EnableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    lbldfc::SetCurrentBrightnessSetting(brightnessOverwrite);
    EXPECT_EQ(brightnessOverwrite, lbldfc::GetCurrentBrightnessSetting());
    EXPECT_EQ(autoAppliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());
}

/**
 * @brief   照度センサー値の入出力で想定する動作をするか（共通）
 */
TEST(StateTransition, AmbientLightSensorValue)
{
    nnt::lbl::LblInitializer initializer;

    float brightnessReference = 0.5f;   // 中クラスの輝度設定値
    float sensorReference = 400.0f;     // 中クラスの照度センサーの値
    float margin = 0.05f;

    // 自動輝度調整offのときは照度センサー値は影響しない。
    lbldfc::SetCurrentBrightnessSetting(brightnessReference);
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::SetAmbientLightSensorValue(sensorReference - 0.1f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 現在の手動輝度調整時は輝度設定値 != 輝度レベルであるが、
    // 近い値とはなるので、マージンを持たせて緩めにテストする。
    EXPECT_LT(brightnessReference - margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
    EXPECT_GT(brightnessReference + margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
    lbldfc::SetAmbientLightSensorValue(sensorReference + 0.1f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_LT(brightnessReference - margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
    EXPECT_GT(brightnessReference + margin, lbldfc::GetBrightnessSettingAppliedToBacklight());
}

/**
* @brief   照度センサー値の入出力で想定する動作をするか（要照度センサ）
*/
TEST(StateTransition, AmbientLightSensorValue_RequireAls)
{
    nnt::lbl::LblInitializer initializer;

    ASSERT_TRUE(lbldfc::IsAmbientLightSensorAvailable());

    float brightnessReference = 0.5f;   // 中クラスの輝度設定値
    float sensorReference = 400.0f;     // 中クラスの照度センサーの値

    // 自動輝度調整on安定時に適用される輝度設定値。
    lbldfc::EnableAutoBrightnessControl();
    float autoAppliedBrightness = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);

    // 写像が単調増加関数である前提をもったテストだが、この前提で問題なし。
    EXPECT_GT(autoAppliedBrightness, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference - 0.1f));
    EXPECT_LT(autoAppliedBrightness, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference + 0.1f));
}

/**
 * @brief   反応遅延の入出力で想定する動作をするか(中間クラスの輝度設定値割り当て時)（要照度センサ）
 */
TEST(StateTransition, ReflectionDelayWithMiddleBrightness_RequireAls)
{
    nnt::lbl::LblInitializer initializer;

    ASSERT_TRUE(lbldfc::IsAmbientLightSensorAvailable());

    float brightnessReference = 0.5f;       // 中間クラスの輝度設定値
    float sensorReference = 816.147339f;    // 最大クラスの照度センサーの値
    // 自動輝度調整on安定時に適用される輝度設定値。
    lbldfc::EnableAutoBrightnessControl();
    float autoAppliedDark = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f);
    float autoAppliedBright = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);
    // 自動輝度調整off時は適用される輝度設定値が照度値に影響されない。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    float appliedBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();

    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());

    ///////////////////////////////////////////////////
    // 自動輝度調整on時
    lbldfc::EnableAutoBrightnessControl();

    // 照度センサーのレベルを 1.0 -> 0.0 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedDarkDelay);

    // 照度センサーのレベルを 0.0 -> 1.0 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedBrightDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightDelay);

    // 照度センサーのレベルを 1.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を少し小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkToDimDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedDarkToDimDelay);

    // 照度センサーのレベルを 0.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を少し大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedBrightToDimDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightToDimDelay);

    ///////////////////////////////////////////////////
    // 細かい仕様まわりを厳しめにチェック

    // 照度センサーの値に応じて輝度が変化する。
    float autoAppliedMiddle = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference / 2.0f);
    EXPECT_LT(autoAppliedDark, autoAppliedMiddle);
    EXPECT_LT(autoAppliedMiddle, autoAppliedBright);
    // 照度センサー値の負の差分が大きいほど暗くなっているか。
    EXPECT_LT(autoAppliedDarkDelay - autoAppliedBright, autoAppliedDarkToDimDelay - autoAppliedBright);
    // 照度センサー値の正の差分が大きいほど明るくなっているか。
    EXPECT_GT(autoAppliedBrightDelay - autoAppliedDark, autoAppliedBrightToDimDelay - autoAppliedDark);
}

/**
 * @brief   反応遅延の入出力で想定する動作をするか(最小クラスの輝度設定値割り当て時)（要照度センサ）
 */
TEST(StateTransition, ReflectionDelayWithLowestBrightness_RequireAls)
{
    nnt::lbl::LblInitializer initializer;
    float margin = 0.05f;

    ASSERT_TRUE(lbldfc::IsAmbientLightSensorAvailable());

    float brightnessReference = 0.0f;       // 最小クラスの輝度設定値
    float sensorReference = 816.147339f;    // 最大クラスの照度センサーの値
    // 自動輝度調整on安定時に適用される輝度設定値。
    lbldfc::EnableAutoBrightnessControl();
    float autoAppliedDark = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f);
    float autoAppliedBright = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);
    // 自動輝度調整off時は適用される輝度設定値が照度値に影響されない。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    float appliedBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();

    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());

    ///////////////////////////////////////////////////
    // 自動輝度調整on時
    lbldfc::EnableAutoBrightnessControl();

    // 照度センサーのレベルを 1.0 -> 0.0 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedDarkDelay);

    // 照度センサーのレベルを 0.0 -> 1.0 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    // 輝度設定値が低いと内部でリミッターがかかって輝度値が変化しない(暗いまま)こともある。
    EXPECT_LE(autoAppliedDark, autoAppliedBrightDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightDelay);

    // 照度センサーのレベルを 1.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を少し小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkToDimDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedDarkToDimDelay);

    // 照度センサーのレベルを 0.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を少し大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    // 輝度設定値が低いと内部でリミッターがかかって輝度値が変化しない(暗いまま)こともある。
    EXPECT_LE(autoAppliedDark, autoAppliedBrightToDimDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightToDimDelay);

    // 輝度設定値に関わらず、照度センサーの値に応じて輝度が変化する。
    float autoAppliedMiddle = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference / 2.0f);
    EXPECT_LT(autoAppliedDark, autoAppliedMiddle);
    EXPECT_LE(autoAppliedMiddle, autoAppliedBright);
    // 輝度設定値が低くても1回程度の更新で多少輝度値は変化する。
    EXPECT_LT(autoAppliedDark, autoAppliedBrightDelay);
    EXPECT_LE(autoAppliedDark, autoAppliedBrightToDimDelay);
    EXPECT_GT(autoAppliedDark + margin, autoAppliedBrightToDimDelay);
    // 照度センサー値の負の差分が大きくても、上記の理由で輝度値の変化量はあまり変わらない。
    EXPECT_LT(autoAppliedDarkDelay - margin, autoAppliedDarkToDimDelay);
    EXPECT_GT(autoAppliedDarkDelay + margin, autoAppliedDarkToDimDelay);
}

/**
 * @brief   反応遅延の入出力で想定する動作をするか(最大クラスの輝度設定値割り当て時)（要照度センサ）
 */
TEST(StateTransition, ReflectionDelayWithHighestBrightness_RequireAls)
{
    nnt::lbl::LblInitializer initializer;
    float margin = 0.05f;

    ASSERT_TRUE(lbldfc::IsAmbientLightSensorAvailable());

    float brightnessReference = 1.0f;       // 最大クラスの輝度設定値
    float sensorReference = 816.147339f;    // 最大クラスの照度センサーの値
    // 自動輝度調整on安定時に適用される輝度設定値。
    lbldfc::EnableAutoBrightnessControl();
    float autoAppliedDark = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f);
    float autoAppliedBright = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference);
    // 自動輝度調整off時は適用される輝度設定値が照度値に影響されない。
    lbldfc::DisableAutoBrightnessControl();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    float appliedBrightness = lbldfc::GetBrightnessSettingAppliedToBacklight();

    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 適用される輝度設定値は変わらない。
    EXPECT_EQ(appliedBrightness, lbldfc::GetBrightnessSettingAppliedToBacklight());

    ///////////////////////////////////////////////////
    // 自動輝度調整on時
    lbldfc::EnableAutoBrightnessControl();

    // 照度センサーのレベルを 1.0 -> 0.0 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(0.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkDelay);
    // 輝度設定値が高いと内部でリミッターがかかって輝度値が変化しない(明るいまま)こともある。
    EXPECT_GE(autoAppliedBright, autoAppliedDarkDelay);

    // 照度センサーのレベルを 0.0 -> 1.0 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedBrightDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightDelay);

    // 照度センサーのレベルを 1.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が明るい状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedBright, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference));
    // 照度センサー値を少し小さくする(暗くする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedDarkToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedDarkToDimDelay);
    // 輝度設定値が高いと内部でリミッターがかかって輝度値が変化しない(明るいまま)こともある。
    EXPECT_GE(autoAppliedBright, autoAppliedDarkToDimDelay);

    // 照度センサーのレベルを 0.0 -> 0.5 にして1回輝度値を更新。
    // 照度センサー値が暗い状態で輝度変化を安定させる。
    EXPECT_EQ(autoAppliedDark, nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, 0.0f));
    // 照度センサー値を少し大きくする(明るくする)。
    lbldfc::SetAmbientLightSensorValue(sensorReference / 2.0f);
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    // 遅延が発生する。
    float autoAppliedBrightToDimDelay = lbldfc::GetBrightnessSettingAppliedToBacklight();
    EXPECT_LT(autoAppliedDark, autoAppliedBrightToDimDelay);
    EXPECT_GT(autoAppliedBright, autoAppliedBrightToDimDelay);

    // 輝度設定値に関わらず、照度センサーの値に応じて輝度が変化する。
    float autoAppliedMiddle = nnt::lbl::GetStableAutoAppliedBrightnessSetting(brightnessReference, sensorReference / 2.0f);
    EXPECT_LT(autoAppliedDark, autoAppliedMiddle);
    EXPECT_LE(autoAppliedMiddle, autoAppliedBright);
    // 輝度設定値が高いと内部でリミッターがかかって1回程度の更新で輝度値はあまり変化しない。
    EXPECT_GT(autoAppliedBright, autoAppliedDarkDelay);
    EXPECT_GE(autoAppliedBright, autoAppliedDarkToDimDelay);
    // 照度センサー値の負の差分が大きくても、上記の理由で輝度値の変化量はあまり変わらない。
    EXPECT_LT(autoAppliedDarkDelay - margin, autoAppliedDarkToDimDelay);
    EXPECT_GT(autoAppliedDarkDelay + margin, autoAppliedDarkToDimDelay);
}

/**
 * @brief   初期化/終了処理が正しくできるか(プロセスで利用するDFCのInitialze/Finalizeを想定)
 */
TEST(StateTransition, InitializeAndFinalize)
{
    // 過剰に終了処理をしても、内部では実行されない。
    for (int i1=0; i1<100; ++i1)
    {
        lbldfc::Finalize();
    }
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::ApplyCurrentBrightnessSettingToBacklight(), "");

    // 初期化すれば問題なし。
    lbldfc::Initialize();
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();

    // 過剰に初期化しても、内部では実行されない。
    for (int i1=0; i1<100; ++i1)
    {
        lbldfc::Initialize();
    }
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();

    // 初期化回数の最大カウントは16回。
    for (int i1=0; i1<15; ++i1)
    {
        lbldfc::Finalize();
    }
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    lbldfc::Finalize();
    EXPECT_DEATH_IF_SUPPORTED(lbldfc::ApplyCurrentBrightnessSettingToBacklight(), "");
}

/**
 * @brief   状態遷移が絡む条件下で点灯/消灯の切り替えが正しくされるか
 */
TEST(StateTransition, SwitchBacklight)
{
    nnt::lbl::LblInitializer initializer;

    // 便宜上、fadeVeryShortTime * 4 == fadeShortTime * 2 == fadeLongTime
    nn::TimeSpan fadeVeryShortTime = nn::TimeSpan::FromMilliSeconds(100LL);
    nn::TimeSpan fadeShortTime = nn::TimeSpan::FromMilliSeconds(200LL);
    nn::TimeSpan fadeLongTime = nn::TimeSpan::FromMilliSeconds(400LL);
    const int margin = 32;
    NN_UNUSED(margin);

    // 一定の遷移速度となる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(127 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(127 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(63 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(63 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(31 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(31 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(31 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(31 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(63 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(63 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(127 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(127 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeVeryShortTime) );
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );

    // 点灯状態かつ消灯開始中に点灯しようとすると遷移時間を上書きできる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_LE(95 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(95 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    // 消灯状態かつ点灯開始中に消灯しようとしても遷移時間を上書きできる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_GE(95 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_LE(95 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    // 点灯状態かつ消灯開始中に消灯(時間を更新)しようとしても遷移時間を上書きできる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeLongTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(95 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_LE(95 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(0, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    // 消灯状態かつ点灯開始中に点灯(時間を更新)しようとしても遷移時間を上書きできる。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_GE(0 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(95 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_GE(95 + margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(fadeLongTime) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    FAKE_ENV_TEST( EXPECT_LE(255 - margin, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeShortTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(255, nn::pwm::GetDuty(nullptr)) );
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    // ApplyCurrentBrightnessSettingToBacklight が来る前に設定しても再設定が可能。
    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    lbldfc::SwitchBacklightOff(0LL);
    lbldfc::SwitchBacklightOn(0LL);
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    lbldfc::SwitchBacklightOn(0LL);
    lbldfc::SwitchBacklightOff(0LL);
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOn(0LL) );
    lbldfc::SwitchBacklightOff(0LL);
    lbldfc::SwitchBacklightOn(fadeLongTime);
    // OnWithStability にとどまらず即座に状態遷移する。
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffInTransitionToOn, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeLongTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(nullptr)) );

    APPLY_BRIGHTNESS_SETTING( lbldfc::SwitchBacklightOff(0LL) );
    lbldfc::SwitchBacklightOn(0LL);
    lbldfc::SwitchBacklightOff(fadeLongTime);
    // OffWithStability にとどまらず即座に状態遷移する。
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    lbldfc::ApplyCurrentBrightnessSettingToBacklight();
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OnInTransitionToOff, lbldfc::GetBacklightSwitchStatus());
    APPLY_BRIGHTNESS_SETTING( nn::os::SleepThread(fadeLongTime) );
    EXPECT_EQ(nn::lbl::BacklightSwitchStatus_OffWithStability, lbldfc::GetBacklightSwitchStatus());
    FAKE_ENV_TEST( EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(nullptr)) );
} // NOLINT(impl/function_size)
