﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ae.h>
#include <nn/oe.h>
#include <nn/applet/applet.h>

#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/la/la_NifmToWifiWebAuthArgumentsWriter.h>
#include <nn/la/la_AppletToNifmArgumentsReader.h>

#include <nn/nifm/nifm_Result.h>

class DummyAppletCallerTest : public ::testing::Test
{
protected:
    virtual void TearDown() {  }
};

TEST_F(DummyAppletCallerTest, WrappedPush_Error)
{
    nn::la::CommonArgumentsWriter commonArgumentsWriter(123);
    commonArgumentsWriter.SetPlayStartupSound(true);

    nn::applet::LibraryAppletHandle libraryAppletHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&libraryAppletHandle, nn::applet::AppletId_LibraryAppletError, nn::applet::LibraryAppletMode_AllForeground));

    commonArgumentsWriter.PushToInChannel(libraryAppletHandle);

    NNT_ASSERT_RESULT_SUCCESS(nn::applet::StartLibraryApplet(libraryAppletHandle));
    nn::applet::JoinLibraryApplet(libraryAppletHandle);

    // 応答を取得
    nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;

    ASSERT_FALSE(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle));

    nn::applet::CloseLibraryApplet(libraryAppletHandle);
}

TEST_F(DummyAppletCallerTest, WrappedPush_WifiWebAuth)
{
    nn::la::CommonArgumentsWriter commonArgumentsWriter(321);
    commonArgumentsWriter.SetPlayStartupSound(true);

    nn::util::Uuid id;
    id.FromString("C842618B-41FC-40AB-9A55-EE35452B39B3");
    uint32_t submitId = 100;
    nn::la::NifmToWifiWebAuthArgumentsWriter wifiWebAuthArgumentsWriter("http://example.com", "http://ctest.cdn.nintendo.net/", id, submitId);

    nn::applet::LibraryAppletHandle libraryAppletHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&libraryAppletHandle, nn::applet::AppletId_LibraryAppletWifiWebAuth, nn::applet::LibraryAppletMode_AllForeground));

    commonArgumentsWriter.PushToInChannel(libraryAppletHandle);
    wifiWebAuthArgumentsWriter.PushToInChannel(libraryAppletHandle);

    nn::applet::StartLibraryApplet(libraryAppletHandle);
    nn::applet::JoinLibraryApplet(libraryAppletHandle);

    // 応答を取得
    nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;

    ASSERT_TRUE(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, appletToNifmArgumentsReader.GetResult());

    nn::applet::CloseLibraryApplet(libraryAppletHandle);
}

TEST_F(DummyAppletCallerTest, DirectPush_Error)
{
    nn::la::CommonArgumentsWriter commonArgumentsWriter(123);
    commonArgumentsWriter.SetPlayStartupSound(true);

    nn::applet::LibraryAppletHandle libraryAppletHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&libraryAppletHandle, nn::applet::AppletId_LibraryAppletError, nn::applet::LibraryAppletMode_AllForeground));

    char buffer[1024];
    ASSERT_GE(sizeof(buffer), commonArgumentsWriter.GetExportSize());
    commonArgumentsWriter.Export(buffer, sizeof(buffer));

    nn::applet::StorageHandle storageHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, commonArgumentsWriter.GetExportSize()));
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, buffer, commonArgumentsWriter.GetExportSize()));
    nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);

    nn::applet::StartLibraryApplet(libraryAppletHandle);
    nn::applet::JoinLibraryApplet(libraryAppletHandle);

    // 応答を取得
    nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;

    ASSERT_FALSE(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle));

    nn::applet::CloseLibraryApplet(libraryAppletHandle);
}

TEST_F(DummyAppletCallerTest, DirectPush_WifiWebAuth)
{
    nn::la::CommonArgumentsWriter commonArgumentsWriter(321);
    commonArgumentsWriter.SetPlayStartupSound(true);

    nn::util::Uuid id;
    id.FromString("C842618B-41FC-40AB-9A55-EE35452B39B3");
    uint32_t submitId = 100;
    nn::la::NifmToWifiWebAuthArgumentsWriter wifiWebAuthArgumentsWriter("http://example.com", "http://ctest.cdn.nintendo.net/", id, submitId);

    nn::applet::LibraryAppletHandle libraryAppletHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&libraryAppletHandle, nn::applet::AppletId_LibraryAppletWifiWebAuth, nn::applet::LibraryAppletMode_AllForeground));

    char buffer1[1024];
    ASSERT_GE(sizeof(buffer1), commonArgumentsWriter.GetExportSize());
    commonArgumentsWriter.Export(buffer1, sizeof(buffer1));

    char buffer2[2048];
    ASSERT_GE(sizeof(buffer2), wifiWebAuthArgumentsWriter.GetExportSize());
    wifiWebAuthArgumentsWriter.Export(buffer2, sizeof(buffer2));

    {
        nn::applet::StorageHandle storageHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, commonArgumentsWriter.GetExportSize()));
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, buffer1, commonArgumentsWriter.GetExportSize()));
        nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);
    }
    {
        nn::applet::StorageHandle storageHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, wifiWebAuthArgumentsWriter.GetExportSize()));
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, buffer2, wifiWebAuthArgumentsWriter.GetExportSize()));
        nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);
    }

    nn::applet::StartLibraryApplet(libraryAppletHandle);
    nn::applet::JoinLibraryApplet(libraryAppletHandle);

    // 応答を取得
    nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;

    ASSERT_TRUE(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, appletToNifmArgumentsReader.GetResult());

    nn::applet::CloseLibraryApplet(libraryAppletHandle);
}

TEST_F(DummyAppletCallerTest, FutureVersion_WifiWebAuth)
{
    // 現状の CommonArguments は、クライアント側から将来のバージョンが送られることは想定していない
    nn::la::CommonArgumentsWriter commonArgumentsWriter(123);
    commonArgumentsWriter.SetPlayStartupSound(true);

    nn::la::detail::NifmToWifiWebAuthArgumentsDataV0 wifiWebAuthArgumentsDataV0;

    nn::util::Strlcpy(wifiWebAuthArgumentsDataV0.authUrl, "http://example.com", sizeof(wifiWebAuthArgumentsDataV0.authUrl));
    nn::util::Strlcpy(wifiWebAuthArgumentsDataV0.connTestUrl, "http://ctest.cdn.nintendo.net/", sizeof(wifiWebAuthArgumentsDataV0.connTestUrl));
    nn::util::Uuid id;
    id.FromString("C842618B-41FC-40AB-9A55-EE35452B39B3");
    wifiWebAuthArgumentsDataV0.profileId = id;
    wifiWebAuthArgumentsDataV0.submitId = 100;

    nn::applet::LibraryAppletHandle libraryAppletHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&libraryAppletHandle, nn::applet::AppletId_LibraryAppletWifiWebAuth, nn::applet::LibraryAppletMode_AllForeground));

    commonArgumentsWriter.PushToInChannel(libraryAppletHandle);

    // wifiWebAuthArguments は、クライアント側から将来のバージョンが送られることを想定
    static const uint32_t WifiWebAuthVersionFuture = 1;
    static const uint32_t DummyFutureParameter = 1234;

    {
        nn::applet::StorageHandle storageHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(WifiWebAuthVersionFuture) + sizeof(wifiWebAuthArgumentsDataV0) + sizeof(DummyFutureParameter)));

        int size = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &WifiWebAuthVersionFuture, sizeof(WifiWebAuthVersionFuture)));
        size += sizeof(WifiWebAuthVersionFuture);
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &wifiWebAuthArgumentsDataV0, sizeof(wifiWebAuthArgumentsDataV0)));
        size += sizeof(wifiWebAuthArgumentsDataV0);
        NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, size, &DummyFutureParameter, sizeof(DummyFutureParameter)));
        nn::applet::PushToInChannel(libraryAppletHandle, storageHandle);
    }

    nn::applet::StartLibraryApplet(libraryAppletHandle);
    nn::applet::JoinLibraryApplet(libraryAppletHandle);

    // 応答を取得（将来バージョンの応答が返る）
    nn::la::AppletToNifmArgumentsReader appletToNifmArgumentsReader;

    ASSERT_TRUE(appletToNifmArgumentsReader.TryPopFromOutChannel(libraryAppletHandle));
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, appletToNifmArgumentsReader.GetResult());

    nn::applet::CloseLibraryApplet(libraryAppletHandle);
}

// nn::ae のメッセージ処理
bool ExpectMessage(nn::ae::Message expectMessage, nn::os::SystemEventType& e) NN_NOEXCEPT
{
    auto message = nn::ae::WaitForNotificationMessage(&e);
    if (message != expectMessage)
    {
        NN_LOG("Unexpected message 0x%08x (expect=0x%08x)\n", message, expectMessage);
        return false;
    }
    return true;
}

nn::ncm::ApplicationId g_ApplicationId;
void RunApplication(nn::ae::SystemAppletParameters* param) NN_NOEXCEPT
{
    // SA の起動
    nn::os::SystemEventType e;
    nn::ae::InitializeNotificationMessageEvent(&e);
    NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoForeground, e));

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}

// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // SystemApplet としてアプリを起動する
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, RunApplication);
}
