﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_UnitHeap.h>
#include "testLMem_Util.h"


namespace {

/**
 * @brief   ユニットヒープのテストで利用される構造体です。
 * @details テスト中に構造体の中身が参照されることはありません。
 */
struct TestNode
{
    int valueA;
    nn::Bit8 valueB[4];
    uintptr_t valueC;
};

/**
 * @brief   テストに与えるパラメータです。
 */
enum UnitHeapTestParam
{
    UnitHeapTestParam_Head,     //!< ヒープの先頭にヒープ管理領域がある
    UnitHeapTestParam_Tail,     //!< ヒープの末尾にヒープ管理領域がある
    UnitHeapTestParam_Outside   //!< ヒープ領域外にヒープ管理領域がある
};

const int g_UnitNum = 100;                // 作成するユニットの数
const size_t g_ThreadStackSize = 8192;
NN_ALIGNAS(4096) char g_ThreadStack[g_ThreadStackSize];
nn::os::ThreadType g_Thread;
nn::os::EventType g_Event;

/**
 * @brief   UnitHeap を繰り返し作成します。
 */
void CreateUnitHeapConstantly(void* arg)
{
    NN_UNUSED(arg);

    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heapAddr;
    nn::lmem::HeapHandle heapHandle;

    heapAddr = std::malloc(HeapSize);

    // 繰り返しヒープを作成
    for(int i = 0; i < 1024; ++i)
    {
        heapHandle = nn::lmem::CreateUnitHeap(heapAddr, HeapSize, 16, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);     // ヒープ作成に成功したか
    }

    nn::lmem::DestroyUnitHeap(heapHandle);
    std::free(heapAddr);

    nn::os::SignalEvent(&g_Event);
}

/**
 * @brief   ユニットヒープのテストで利用するテストフィクスチャです。
 */
class UnitHeapBasicTest : public testing::TestWithParam<int>
{
protected:

    const size_t m_HeapSize;                // グローバルヒープのサイズ
    const int m_MinAlignment;               // ユニットヒープの最小アライメント
    uintptr_t m_Heap;                       // グローバルヒープのポインタ
    size_t m_UnitHeapSize;                  // ユニットヒープとして利用するサイズ
    nn::lmem::HeapHandle m_HeapHandle;      // ユニットヒープのヒープハンドル
    nn::lmem::HeapCommonHead m_HeapHead;    // ヒープ管理領域（ヒープ共通ヘッダ）
    int m_TestParam;                        // テスト時に渡されるパラメータ（UnitHeapTestParam列挙型）
    size_t m_HeadSize;                      // ヒープ内部に確保されている管理領域のサイズ

    UnitHeapBasicTest() : m_HeapSize(nn::os::MemoryBlockUnitSize), m_MinAlignment(4){}

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        m_UnitHeapSize = nn::lmem::GetRequiredUnitHeapSize(sizeof(TestNode), g_UnitNum, m_MinAlignment, true);
        m_Heap = reinterpret_cast<uintptr_t>(std::malloc(m_HeapSize));
    }

    /**
     * @brief テスト開始時に毎回呼び出される関数です。
     */
    void SetUpParam()
    {
        m_TestParam = GetParam();
        if( m_TestParam == UnitHeapTestParam_Outside)
        {
            m_HeadSize = 0;
            m_UnitHeapSize = nn::lmem::GetRequiredUnitHeapSize(sizeof(TestNode), g_UnitNum, m_MinAlignment, false);
        }
        else
        {
            m_HeadSize = sizeof(nn::lmem::HeapCommonHead);
        }
    }

    /**
     * @brief m_TestParam の値に応じて設定を変えてユニットヒープの作成をします。
     */
    void Create(size_t heapSize, size_t unitSize, int option, int alignment)
    {
        if(m_TestParam == UnitHeapTestParam_Head)
        {
            m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), heapSize, unitSize, option, alignment, nn::lmem::InfoPlacement_Head);
        }
        else if(m_TestParam == UnitHeapTestParam_Tail)
        {
            m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), heapSize, unitSize, option, alignment, nn::lmem::InfoPlacement_Tail);
        }
        else if(m_TestParam == UnitHeapTestParam_Outside)
        {
            m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), heapSize, unitSize, option, alignment, &m_HeapHead);
        }
        ASSERT_NOT_NULL(m_HeapHandle);     // ヒープ作成に成功したか
    }

    /**
     * @brief テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        nn::lmem::InvalidateUnitHeap(m_HeapHandle);
        nn::lmem::DestroyUnitHeap(m_HeapHandle);       // ヒープの破棄
        std::free(reinterpret_cast<void*>(m_Heap));    // グローバルヒープへの返却
    }

    /**
     * @brief ユニットヒープ全体のユニット数と、確保済みのユニット、フリーのユニットの数に齟齬がないことを確認します。
     */
    bool CheckAllocatedCount()
    {
        return (static_cast<int>((nn::lmem::GetTotalSize(m_HeapHandle) - m_HeadSize) / nn::lmem::GetUnitHeapUnitSize(m_HeapHandle) - nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle))) == nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle);
    }
};



}   // unnamed namespace

// =============
// テスト
// =============

// 単純なヒープの作成
// ヒープを作成します。ヒープの作成が成功（ハンドルが nullptr でない）なら OK
TEST_F(UnitHeapBasicTest, SimpleCreateHeap)
{
    // ヒープの作成
    m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), m_UnitHeapSize, sizeof(TestNode), nn::lmem::CreationOption_NoOption, m_MinAlignment, nn::lmem::InfoPlacement_Head);
    ASSERT_NOT_NULL(m_HeapHandle);     // ヒープ作成に成功したか
    // 確保したユニット数に齟齬が発生していない
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), g_UnitNum);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), 0);
}

// ヒープの作成とメモリブロックの確保
// メモリブロックの確保が、指定したユニットサイズで正常に行われていれば、 OK
TEST_P(UnitHeapBasicTest, CreateAndAllocate)
{
    void* pMemoryAddresses[g_UnitNum];

    SetUpParam();

    // ヒープの作成
    Create(m_UnitHeapSize, sizeof(TestNode), nn::lmem::CreationOption_NoOption, m_MinAlignment);

    // 確保したユニット数に齟齬が発生していない
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), g_UnitNum);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), 0);

    EXPECT_TRUE(CheckAllocatedCount());

    // 全ユニットを確保済みにする
    for(int i = 0; i < g_UnitNum; ++i)
    {
        // 確保したユニット数に齟齬が発生していない
        EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), g_UnitNum - i);
        EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), i);
        EXPECT_TRUE(CheckAllocatedCount());
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        pMemoryAddresses[i] = p;
        for(int j = 0; j < i - 1; ++j)
        {
            EXPECT_NE(p, pMemoryAddresses[j]);  // 以前渡した領域と同じ領域を渡していない
        }
    }

    // 確保したユニット数に齟齬が発生していない
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), 0);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), g_UnitNum);
    EXPECT_TRUE(CheckAllocatedCount());
    // 全ユニットが確保済みの場合は、確保に失敗する
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));

    // どのユニットも正常に解放できる
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[g_UnitNum - 1]);
    {
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        pMemoryAddresses[g_UnitNum - 1] = p;
    }

    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[0]);
    {
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        pMemoryAddresses[0] = p;
    }

    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[g_UnitNum / 2]);
    {
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        pMemoryAddresses[g_UnitNum / 2] = p;
    }

    // 全てのユニットの解放
    for(int i = 0; i < g_UnitNum; ++i)
    {
        nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[i]);
    }

    // 確保／解放を何度も繰り返しても失敗しない
    for(int i = 0; i < 10000; ++i)
    {
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        nn::lmem::FreeToUnitHeap(m_HeapHandle, p);
    }
}

// ユニットヒープサイズのチェック
// ユニットヒープのサイズ取得関数が正しい値を返しているなら、OK
TEST_P(UnitHeapBasicTest, Size)
{
    SetUpParam();

    m_UnitHeapSize = 256 + m_HeadSize;

    const size_t UnitSize = 128;    // 1ユニットのサイズ
    void* pMemoryAddress;

    // ヒープの作成
    Create(m_UnitHeapSize, UnitSize, nn::lmem::CreationOption_NoOption, m_MinAlignment);

    // サイズのテスト
    size_t allocatableCount = nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle);
    size_t totalSize = nn::lmem::GetTotalSize(m_HeapHandle);
    EXPECT_EQ(allocatableCount * UnitSize + m_HeadSize, m_UnitHeapSize);
    EXPECT_EQ(totalSize, m_UnitHeapSize);

    // メモリブロックの確保
    pMemoryAddress = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);

    // サイズのテスト
    allocatableCount = nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle);
    EXPECT_EQ(totalSize - m_HeadSize, allocatableCount * UnitSize + UnitSize);

    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddress);
}

// ヒープ作成の繰り返し
// ヒープを同じアドレスで繰り返し呼んでも問題ないことを確認する
TEST(UnitHeapCreateTest, CreateHeapConstantly)
{
    int threadPriority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread());
    if(threadPriority < nn::os::LowestThreadPriority)
    {
        threadPriority++;
    }
    auto result = nn::os::CreateThread(&g_Thread, CreateUnitHeapConstantly, NULL, g_ThreadStack, g_ThreadStackSize, threadPriority);
    ASSERT_TRUE(result.IsSuccess());

    nn::os::InitializeEvent(&g_Event, false, nn::os::EventClearMode_AutoClear);

    const int64_t timeOut = 20;     // 20 秒でタイムアウトとする
    nn::os::StartThread(&g_Thread);

    ASSERT_TRUE(nn::os::TimedWaitEvent(&g_Event, nn::TimeSpan::FromSeconds(timeOut)));

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
    nn::os::FinalizeEvent(&g_Event);
}

// ヒープの繰り返し前の確保状況のクリア
// Create 後に以前のヒープ状況がクリアされていることを確認する
TEST_F(UnitHeapBasicTest, CreateHeapClear)
{
    m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), m_UnitHeapSize, 16, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);
    int allocatableCount = nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle);

    void* ptr = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
    ASSERT_NOT_NULL(ptr);
    EXPECT_GT(allocatableCount, nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle));

     // Destroy せずに再度 Create
    m_HeapHandle = nn::lmem::CreateUnitHeap(reinterpret_cast<void*>(m_Heap), m_UnitHeapSize, 16, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(m_HeapHandle);
    // ヒープは初期化されているはず
    EXPECT_EQ(allocatableCount, nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle));

    nn::lmem::DestroyUnitHeap(m_HeapHandle);
}

// =============
// テスト（Debug ビルド、Develop ビルドでのみ有効）
// =============

// メモリ確保時のオプションとメモリフィルは、Release ビルド以外の時のみ有効になる
#if defined(NN_TEST_MEM_DEBUG)

// デバッグフィル
// ヒープ作成、確保、解放時に指定した値でフィルされていれば、OK
TEST_P(UnitHeapBasicTest, DebugFill)
{
    SetUpParam();

    m_UnitHeapSize = 256 + m_HeadSize;

    const size_t UnitSize = 128;    // 1ユニットのサイズ
    const nn::Bit32 FillUnallocated = 0x01234568;
    const nn::Bit32 FillAllocate = 0x456789ac;
    const nn::Bit32 FillFree  = 0x89abcdee;
    void* pMemoryAddress;

    // DebugFill で埋める値を設定
    nn::lmem::SetFillValue(nn::lmem::FillType_Unallocated, FillUnallocated);
    nn::lmem::SetFillValue(nn::lmem::FillType_Allocate, FillAllocate);
    nn::lmem::SetFillValue(nn::lmem::FillType_Free, FillFree);

    // DebugFill で埋める値を確認
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Unallocated), FillUnallocated);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Allocate), FillAllocate);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Free), FillFree);

    // ヒープの作成
    Create(m_UnitHeapSize, UnitSize, nn::lmem::CreationOption_DebugFill, m_MinAlignment);

    // DebugFill (未使用時) テスト
    if(m_TestParam == UnitHeapTestParam_Head)
    {
        pMemoryAddress = reinterpret_cast<void*>((m_Heap) + m_HeadSize + sizeof(nn::lmem::detail::UnitHead));
    }
    else
    {
        pMemoryAddress = reinterpret_cast<void*>((m_Heap) + sizeof(nn::lmem::detail::UnitHead));
    }

    for(int i = 0; i < nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle); ++i)
    {
        // 次の空きブロックへのポインタの部分以外がフィルされているかチェック
        void* pAddress = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + UnitSize * i);
        EXPECT_TRUE(HasEqualNum(pAddress, UnitSize - sizeof(nn::lmem::detail::UnitHead), FillUnallocated));
    }

    // DebugFill (確保時) テスト
    pMemoryAddress = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, UnitSize, FillAllocate));

    // DebugFill (解放時) テスト
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddress);
    // 次の空きブロックへのポインタの部分以外がフィルされているかチェック
    void* pAddress = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + sizeof(nn::lmem::detail::UnitHead));
    EXPECT_TRUE(HasEqualNum(pAddress, UnitSize - sizeof(nn::lmem::detail::UnitHead), FillFree));
}

// 確保時のゼロクリア
// メモリ確保時にメモリが 0 でフィルされていれば、OK
TEST_P(UnitHeapBasicTest, ZeroClear)
{
    SetUpParam();

    m_UnitHeapSize = 256 + m_HeadSize;

    const size_t UnitSize = 128;    // 1ユニットのサイズ
    void* pMemoryAddress;

    // ヒープの作成
    Create(m_UnitHeapSize, UnitSize, nn::lmem::CreationOption_ZeroClear, m_MinAlignment);

    // ZeroClear テスト (確保時)
    pMemoryAddress = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, UnitSize, 0));

    // Read/Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, UnitSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, UnitSize));

    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddress);
}

// Dump
// ダンプした内容が、こちらの意図した通りであれば、OK
TEST_P(UnitHeapBasicTest, Dump)
{
    SetUpParam();

    void* pMemoryAddresses[g_UnitNum];

    // ヒープの作成
    Create(m_UnitHeapSize, sizeof(TestNode), nn::lmem::CreationOption_NoOption, m_MinAlignment);

    // 確保したユニット数に齟齬が発生していない
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), g_UnitNum);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), 0);

    EXPECT_TRUE(CheckAllocatedCount());

    // 全てのユニットが空き状態のダンプ結果になるはず
    nn::lmem::DumpUnitHeap(m_HeapHandle);
    NN_LOG("Visual Check: Are all units empty?\n\n");

    for(int i = 0; i < g_UnitNum; ++i)
    {
        EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), g_UnitNum - i);
        EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), i);
        void* p = nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
        ASSERT_NOT_NULL(p);
        pMemoryAddresses[i] = p;
        for(int j = 0; j < i; ++j)
        {
            EXPECT_NE(p, pMemoryAddresses[j]);
        }
    }
    // ユニットが全部消費されている
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), 0);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatedCount(m_HeapHandle), g_UnitNum);
    // 全ユニットが確保済みの場合は、確保に失敗する
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));
    ASSERT_NULL(nn::lmem::AllocateFromUnitHeap(m_HeapHandle));

    // ユニットが全部消費されたダンプ結果になるはず
    nn::lmem::DumpUnitHeap(m_HeapHandle);
    NN_LOG("Visual Check: Are all units used?\n\n");

    // 5つ分 Free する
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[0]);
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[20]);
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[40]);
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[70]);
    nn::lmem::FreeToUnitHeap(m_HeapHandle, pMemoryAddresses[99]);
    EXPECT_EQ(nn::lmem::GetUnitHeapAllocatableCount(m_HeapHandle), 5);

    // ユニットが95%消費されたダンプ結果になるはず
    nn::lmem::DumpUnitHeap(m_HeapHandle);
    NN_LOG("Visual Check: Are 95 units used?\n\n");
}

#endif  // #if defined(NN_TEST_MEM_DEBUG)

INSTANTIATE_TEST_CASE_P(HeapHeadPosition,
                        UnitHeapBasicTest,
                        testing::Values(UnitHeapTestParam_Head, UnitHeapTestParam_Tail, UnitHeapTestParam_Outside));

