﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <limits>
#include <nn/nn_Common.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/kpr/kpr_KeyCode.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>
#include <nnt/nntest.h>

namespace {

::nn::kpr::KeyCodeMap g_KeyCodeMap;

} // namespace

//!< 指定された変換モードをサポートするか否かが正しく処理されるか (EnglishUs)
TEST(KeyCodeSuite, IsKeyCodeMapModeSupported1)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie));
}

//!< 指定された変換モードをサポートするか否かが正しく処理されるか (Japanese)
TEST(KeyCodeSuite, IsKeyCodeMapModeSupported2)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie));
}

//!< 指定された変換モードをサポートするか否かが正しく処理されるか (Russian)
TEST(KeyCodeSuite, IsKeyCodeMapModeSupported3)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie));
}

//!< 指定された変換モードをサポートするか否かが正しく処理されるか (Korean)
TEST(KeyCodeSuite, IsKeyCodeMapModeSupported4)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie));
}

//!< 指定された変換モードをサポートするか否かが正しく処理されるか (ChineseTraditional)
TEST(KeyCodeSuite, IsKeyCodeMapModeSupported5)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic));
    EXPECT_FALSE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin));
    EXPECT_TRUE(
        ::nn::kpr::IsKeyCodeMapModeSupported(
            g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie));
}

//!< Usage Id の境界値が正しく処理されるか
TEST(KeyCodeSuite, UsageIdTest1)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = static_cast<int>(::std::numeric_limits<uint16_t>::min());
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index - 1;
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = static_cast<int>(::std::numeric_limits<uint16_t>::max());
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< Alt Gr キーの押下が正しく判定されるか
TEST(KeyCodeSuite, AltGrTest1)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E1, keyCode.code);
}

//!< Shift キーの押下が正しく判定されるか
TEST(KeyCodeSuite, ShiftTest1)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::A::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::Gui>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest1)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00A3, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0023, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest2)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::U::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00FA, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::U::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00DA, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest3)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest4)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest5)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00C1, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D9::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0039, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D9::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0028, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest6)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}

//!< 英語 (イギリス) のレイアウトにおいてハングルマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedKingdomTest7)
{
    auto language = ::nn::settings::Language_BritishEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest1)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0060, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D2::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0040, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest2)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest3)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest4)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest5)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0043, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0060, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x007E, keyCode.code);
}

//!< 英語 (アメリカ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesTest6)
{
    auto language = ::nn::settings::Language_AmericanEnglish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 英語 (US-International) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest1)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x007A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0303, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 英語 (US-International) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest2)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::T::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00FE, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::T::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00DE, keyCode.code);
}

//!< 英語 (US-International) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest3)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (US-International) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest4)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 英語 (US-International) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest5)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00C7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00AB, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Comma::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003C, keyCode.code);
}

//!< 英語 (US-International) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, EnglishUnitedStatesInternationalTest6)
{
    auto language = ::nn::settings::Language_Dutch;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}

//!< フランス語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest1)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x006D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0031, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< フランス語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest2)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00A4, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< フランス語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest3)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0071, keyCode.code);
}

//!< フランス語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest4)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0071, keyCode.code);
}

//!< フランス語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest5)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0051, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x20AC, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003E, keyCode.code);
}

//!< フランス語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest6)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< フランス語のレイアウトにおいて注音マップが正しく処理されるか
TEST(KeyCodeSuite, FrenchTest7)
{
    auto language = ::nn::settings::Language_French;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0071, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest1)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E9, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00C9, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest2)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::P::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00B6, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::P::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest3)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest4)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest5)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x004F, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::O::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00A7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0031, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D1::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0021, keyCode.code);
}

//!< フランス語 (カナダ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, FrenchCanadaTest6)
{
    auto language = ::nn::settings::Language_CanadianFrench;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< ドイツ語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, GermanTest1)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0079, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Y::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0023, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ドイツ語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, GermanTest2)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0040, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ドイツ語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, GermanTest3)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ドイツ語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, GermanTest4)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ドイツ語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, GermanTest5)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::M::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x004D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::M::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00B5, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0301, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0300, keyCode.code);
}

//!< ドイツ語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, GermanTest6)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< ドイツ語のレイアウトにおいて倉頡マップが正しく処理されるか
TEST(KeyCodeSuite, GermanTest7)
{
    auto language = ::nn::settings::Language_German;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< イタリア語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest1)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E8, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E9, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00F9, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< イタリア語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest2)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005B, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x007B, keyCode.code);
}

//!< イタリア語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest3)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< イタリア語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest4)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< イタリア語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest5)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Z::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D5::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x20AC, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D0::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0030, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D0::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);
}

//!< イタリア語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ItalianTest6)
{
    auto language = ::nn::settings::Language_Italian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}

//!< 日本語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest1)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::ZenkakuHankaku::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Yen::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Muhenkan::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 日本語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest2)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 日本語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest4)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3042, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3042, keyCode.code);
}

//!< 日本語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest6)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x30A4, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x30A3, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x30A4, keyCode.code);
}

//!< 日本語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest7)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0043, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002A, keyCode.code);
}

//!< 日本語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest8)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 日本語のレイアウトにおいて Num Lock キーがかな入力時に正しく処理されるか
TEST(KeyCodeSuite, JapaneseTest9)
{
    auto language = ::nn::settings::Language_Japanese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    *reinterpret_cast<uint32_t*>(&g_KeyCodeMap) |= 0x00000006;

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}

//!< ポルトガル語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest1)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00C7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0303, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ポルトガル語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest2)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0308, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ポルトガル語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest3)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ポルトガル語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest4)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ポルトガル語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest5)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0045, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x20AC, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D8::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0038, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D8::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0028, keyCode.code);
}

//!< ポルトガル語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, PortugueseTest6)
{
    auto language = ::nn::settings::Language_Portuguese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest1)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x044A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x042A, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest2)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest3)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest4)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest5)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0424, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002E, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002C, keyCode.code);
}

//!< ロシア語 (キリル) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, RussianCyrillicTest6)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cyrillic);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest1)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backslash::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0023, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest2)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest3)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest4)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest5)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002F, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Slash::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003F, keyCode.code);
}

//!< ロシア語 (ラテン) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, RussianLatinTest6)
{
    auto language = ::nn::settings::Language_Russian;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< スペイン語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest1)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00F1, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Semicolon::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00D1, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00E7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< スペイン語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest2)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005B, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::OpenBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< スペイン語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest3)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< スペイン語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest4)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< スペイン語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest5)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0045, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x20AC, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00A1, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Plus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x00BF, keyCode.code);
}

//!< スペイン語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, SpanishTest6)
{
    auto language = ::nn::settings::Language_Spanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest1)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0071, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0051, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x007D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest2)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Minus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Minus::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest3)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest4)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest5)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0051, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0040, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002B, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::CloseBracket::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x002A, keyCode.code);
}

//!< スペイン語 (ラテンアメリカ) のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, SpanishLatinAmericaTest6)
{
    auto language = ::nn::settings::Language_LatinAmericanSpanish;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 韓国語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest1)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 韓国語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest2)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 韓国語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest3)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 韓国語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest4)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 韓国語のレイアウトにおいてハングルマップが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest5)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3142, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3143, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Q::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x3142, keyCode.code);
}

//!< 韓国語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest7)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0043, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0027, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0022, keyCode.code);
}

//!< 韓国語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, KoreanTest8)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 韓国語のレイアウトにおいて Num Lock キーがハングル入力時に正しく処理されるか
TEST(KeyCodeSuite, KoreanTest9)
{
    auto language = ::nn::settings::Language_Korean;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}

//!< 簡体字中国語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest1)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0060, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D2::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0040, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Tilde::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest2)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::RightAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest3)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest4)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいてハングルマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest5)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest6)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0043, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0060, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Backquote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x007E, keyCode.code);
}

//!< 簡体字中国語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ChineseSimplifiedTest7)
{
    auto language = ::nn::settings::Language_SimplifiedChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 繁体字中国語のレイアウトが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest1)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0041, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Pipe::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x005C, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x003D, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::NumPadEquals::Index + 1;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0000, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて Alt Gr マップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest2)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Control>();
    modifiers.Set<::nn::hid::KeyboardModifier::LeftAlt>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいてひらがなマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest3)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hiragana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいてカタカナマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest4)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Katakana);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいてハングルマップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest5)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::A::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Hangul);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0061, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて注音マップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest6)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x02C7, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0023, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::D3::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x02C7, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて倉頡マップが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest7)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x6C34, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0045, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::E::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_FALSE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x6C34, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて Caps Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest8)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = int();
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0043, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::C::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0063, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0027, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    usageId = ::nn::hid::KeyboardKey::Quote::Index;
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::CapsLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0022, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて Num Lock キーが正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest9)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::Shift>();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Default);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);
}

//!< 繁体字中国語のレイアウトにおいて Num Lock キーが注音入力時に正しく処理されるか
TEST(KeyCodeSuite, ChineseTraditionalTest10)
{
    auto language = ::nn::settings::Language_TraditionalChinese;
    auto languageCode = ::nn::settings::LanguageCode::Make(language);
    ::nn::settings::system::SetLanguageCode(languageCode);
    ::std::memset(&g_KeyCodeMap, 0, sizeof(g_KeyCodeMap));
    ::nn::kpr::GetKeyCodeMap(&g_KeyCodeMap);

    auto keyCode = ::nn::kpr::KeyCode();
    auto usageId = ::nn::hid::KeyboardKey::NumPad5::Index;
    auto modifiers = ::nn::hid::KeyboardModifierSet();

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Zhuyin);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0020, keyCode.code);

    keyCode = ::nn::kpr::KeyCode();
    modifiers.Reset();
    modifiers.Set<::nn::hid::KeyboardModifier::NumLock>();
    ::nn::kpr::GetKeyCode(
        &keyCode, usageId, modifiers,
        g_KeyCodeMap, ::nn::kpr::KeyCodeMapMode_Cangjie);
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsPrintable>());
    EXPECT_TRUE(
        keyCode.attributes.Test<
            ::nn::kpr::KeyCodeAttribute::IsAsciiConvertible>());
    EXPECT_EQ(0x0035, keyCode.code);
}
