﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <utility>
#include <memory>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/init.h>

#include "../Common/IProtoJitLoader.sfdl.h"
#include "../Common/TestVmJitAarch64.h"
#include "../Common/Utility.h"

namespace nn { namespace jit { namespace testvm {

class JitCompileEnvironmentImpl
{
private:

    static Result MapProcessMemory(uintptr_t* pOut, svc::Handle processHandle, uint64_t codeAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        jit::AslrAllocator aslrAllocator(svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    retry:
        auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
        NN_RESULT_TRY(svc::MapProcessMemory(toAddr, processHandle, codeAddr, codeSize))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
        NN_RESULT_END_TRY
        *pOut = toAddr;
        NN_RESULT_SUCCESS;
    }

public:

    Result Initialize(sf::ISharedObject* pParent, svc::Handle processHandle, uint64_t codeAddress, uint64_t codeSize) NN_NOEXCEPT
    {
        uintptr_t codeDataAddress;
        NN_RESULT_DO(MapProcessMemory(&codeDataAddress, processHandle, codeAddress, codeSize));

        std::memset(reinterpret_cast<void*>(codeDataAddress), 0, codeSize);
        StoreDataCacheAndMemoryBarrier(codeDataAddress, codeSize);
        NN_RESULT_DO(svc::SetProcessMemoryPermission(processHandle, codeAddress, codeSize, svc::MemoryPermission_ReadExecute));
        this->m_pParent = sf::SharedPointer<sf::ISharedObject>(pParent, true);
        this->m_ProcessHandle = processHandle;
        this->m_CodeAddress = codeAddress;
        this->m_CodeSize = codeSize;
        this->m_CodeDataAddress = codeDataAddress;
        NN_RESULT_SUCCESS;
    }

    Result GetCodeAddress(sf::Out<uint64_t> pOut) NN_NOEXCEPT
    {
        *pOut = m_CodeAddress;
        NN_RESULT_SUCCESS;
    }

    Result ShareRom(uint64_t tag, sf::NativeHandle&& transferMemoryHandle, uint64_t size) NN_NOEXCEPT
    {
        auto success = false;

        NN_RESULT_DO(m_SharedRom.Initialize(std::move(transferMemoryHandle), size));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                m_SharedRom.Release();
            }
        };

        NN_RESULT_DO(OnShareRom(tag, reinterpret_cast<const void*>(m_SharedRom.GetAddress()), m_SharedRom.GetSize()));

        success = true;
        NN_RESULT_SUCCESS;
    }

protected:

    virtual ~JitCompileEnvironmentImpl() NN_NOEXCEPT
    {
        if (m_CodeDataAddress)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(m_ProcessHandle, m_CodeAddress, m_CodeSize, svc::MemoryPermission_ReadExecute));
            svc::UnmapProcessMemory(m_CodeDataAddress, m_ProcessHandle, m_CodeAddress, m_CodeSize);
        }
    }

    // virtual
    virtual Result OnShareRom(uint64_t tag, const void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(tag);
        NN_UNUSED(p);
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    uintptr_t GetCodeDataAddress() const NN_NOEXCEPT
    {
        return m_CodeDataAddress;
    }

    uint64_t GetCodeAddress() const NN_NOEXCEPT
    {
        return m_CodeAddress;
    }

private:

    sf::SharedPointer<sf::ISharedObject> m_pParent;
    svc::Handle m_ProcessHandle = {};
    uintptr_t m_CodeDataAddress = 0;
    uint64_t m_CodeAddress = 0;
    uint64_t m_CodeSize = 0;

    class SharedRom
    {
    public:

        Result Initialize(sf::NativeHandle&& transferMemoryHandle, uint64_t size) NN_NOEXCEPT
        {
            auto success = false;

            auto handle = svc::Handle(transferMemoryHandle.GetOsHandle());
            jit::AslrAllocator aslrAllocator(svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        retry:
            auto addr = aslrAllocator.GetAslrRegion(size);
            NN_RESULT_TRY(svc::MapTransferMemory(handle, addr, size, svc::MemoryPermission_Read))
                NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
                {
                    goto retry;
                }
            NN_RESULT_END_TRY
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapTransferMemory(handle, addr, size));
                }
            };

            success = true;
            this->m_Handle = std::move(transferMemoryHandle);
            this->m_Address = addr;
            this->m_Size = size;
            NN_RESULT_SUCCESS;
        }

        ~SharedRom()
        {
            Release();
        }

        void Release() NN_NOEXCEPT
        {
            if (m_Address)
            {
                auto handle = svc::Handle(m_Handle.GetOsHandle());
                NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapTransferMemory(handle, m_Address, m_Size));
                this->m_Address = 0;
                this->m_Size = 0;
                m_Handle.Reset();
            }
        }

        uintptr_t GetAddress() const NN_NOEXCEPT
        {
            return m_Address;
        }

        size_t GetSize() const NN_NOEXCEPT
        {
            return m_Size;
        }

    private:

        sf::NativeHandle m_Handle = {};
        uintptr_t m_Address = 0;
        size_t m_Size = 0;

    };
    SharedRom m_SharedRom;
};

class TestVmJitCompileEnvironmentImpl
    : public JitCompileEnvironmentImpl
{
public:

    Result SetupTestVmCompiler() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result GenerateCode(sf::Out<uint64_t> pEntry, sf::Out<uint64_t> pCodeSize, sf::Out<uint32_t> pInstructionCount, int32_t pc) NN_NOEXCEPT
    {
        size_t codeSize;
        uint32_t instructionCount;
        auto codeOffset = m_CurrentCodeOffset;
        auto codeData = reinterpret_cast<char*>(GetCodeDataAddress()) + codeOffset;

        JitRunner::GenerateCodeImpl(&codeSize, &instructionCount, codeData, m_Instructions, pc);
        StoreDataCacheAndMemoryBarrier(reinterpret_cast<uintptr_t>(codeData), codeSize);
        this->m_CurrentCodeOffset += codeSize;

        *pEntry = GetCodeAddress() + codeOffset;
        *pCodeSize = codeSize;
        *pInstructionCount = instructionCount;
        NN_RESULT_SUCCESS;
    }

private:

    virtual Result OnShareRom(uint64_t tag, const void* p, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(tag);
        this->m_Instructions = static_cast<const Instruction*>(p);
        this->m_InstructionCount = size / sizeof(*m_Instructions);
        NN_RESULT_SUCCESS;
    }

    const Instruction* m_Instructions = nullptr;
    size_t m_InstructionCount = 0;
    int64_t m_CurrentCodeOffset = 0;

};

class JitEnvironmentImpl
    : public sf::ISharedObject
{
private:

    static Result MapProcessCodeMemory(uint64_t* pOut, svc::Handle processHandle, uint64_t bufferAddr, uint64_t codeSize) NN_NOEXCEPT
    {
        jit::AslrAllocator aslrAllocator(processHandle);
    retry:
        auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
        NN_RESULT_TRY(svc::MapProcessCodeMemory(processHandle, toAddr, bufferAddr, codeSize))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
        NN_RESULT_END_TRY
        *pOut = toAddr;
        NN_RESULT_SUCCESS;
    }

public:

    Result Initialize(sf::NativeHandle&& processHandle, Bit64 processId, uint64_t bufferAddress, uint64_t codeSize) NN_NOEXCEPT
    {
        auto handle = svc::Handle(processHandle.GetOsHandle());

        uint64_t codeAddress;
        NN_RESULT_DO(MapProcessCodeMemory(&codeAddress, handle, bufferAddress, codeSize));

        this->m_ProcessHandle = std::move(processHandle);
        this->m_BufferAddress = bufferAddress;
        this->m_CodeAddress = codeAddress;
        this->m_CodeSize = codeSize;
        NN_RESULT_SUCCESS;
    }

    virtual ~JitEnvironmentImpl() NN_NOEXCEPT
    {
        if (m_CodeAddress)
        {
            NN_SDK_ASSERT(m_ProcessHandle.GetOsHandle());
            NN_SDK_ASSERT(m_BufferAddress);
            auto result = svc::UnmapProcessCodeMemory(svc::Handle(m_ProcessHandle.GetOsHandle()), m_CodeAddress, m_BufferAddress, m_CodeSize);
            NN_UNUSED(result);
        }
    }

    Result CreateTestVmJitCompileEnvironment(sf::Out<sf::SharedPointer<ITestVmJitCompileEnvironment>> pOut) NN_NOEXCEPT
    {
        auto p = sf::CreateSharedObjectEmplaced<ITestVmJitCompileEnvironment, TestVmJitCompileEnvironmentImpl>();
        NN_RESULT_DO(p.GetImpl().Initialize(this, svc::Handle(m_ProcessHandle.GetOsHandle()), m_CodeAddress, m_CodeSize));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

private:

    sf::NativeHandle m_ProcessHandle;
    uint64_t m_BufferAddress = 0;
    uint64_t m_CodeAddress = 0;
    uint64_t m_CodeSize = 0;

};

class ProtoJitLoader2Impl
{
public:

    Result CreateJitEnvironment(sf::Out<sf::SharedPointer<IJitEnvironment>> pOut, sf::NativeHandle&& processHandle, Bit64 processId, uint64_t bufferAddress, uint64_t bufferSize) NN_NOEXCEPT
    {
        auto handle = svc::Handle(processHandle.GetOsHandle());

        Bit64 processHandleProcessId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::GetProcessId(&processHandleProcessId, handle));
        NN_ABORT_UNLESS(processHandleProcessId == processId, "TODO: Result");

        auto p = sf::CreateSharedObjectEmplaced<IJitEnvironment, JitEnvironmentImpl>();
        NN_RESULT_DO(p.GetImpl().Initialize(std::move(processHandle), processId, bufferAddress, bufferSize));

        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

};

}}}

namespace {

    using namespace nn;

    struct MyHipcServerManagerOption
    {
        static const bool CanDeferInvokeRequest = false;
        static const int ObjectInSubDomainCountMax = 0;
        static const size_t PointerTransferBufferSize = 0;
        static const int SubDomainCountMax = 0;
    };

    class MyHipcServerManager
        : public sf::HipcSimpleAllInOneServerManager<10, 1, MyHipcServerManagerOption> // セッション数等は、後程調整
    {
    };

    MyHipcServerManager g_MyHipcServerManager;

}

extern "C" void nninitStartup()
{
    static char g_Buffer[1024 * 1024];
    nn::init::InitializeAllocator(g_Buffer, sizeof(g_Buffer));
}

extern "C" void nnMain()
{
    using namespace nn::jit::testvm;
    auto p = sf::CreateSharedObjectEmplaced<IProtoJitLoader2, ProtoJitLoader2Impl>();
    g_MyHipcServerManager.RegisterObjectForPort(std::move(p), 10, "jit:p2");
    g_MyHipcServerManager.LoopAuto();
}
