﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "MomentModeState.h"
#include "ImageTransferModeState.h"

#include "ReflectionCheckerView.h"

#include <nn/time.h>
#include <sstream>

void DrawCameraParameters(const std::vector<MenuItem>& menu,
    const float offsetX, const float offsetY,
    int selectIndex,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);

    for (size_t index = 0; index < menu.size(); index++)
    {
        pTextWriter->SetTextColor(((size_t) selectIndex == index) ? Red : White);
        pTextWriter->SetCursor(offsetX, offsetY + index * 30.0f);

        std::stringstream text;
        text << menu[index].GetName() << ": ";
        menu[index].Read(text);

        pTextWriter->Print(text.str().c_str());
    }
}

void DrawRecordFinishedNotification(
    const float offsetX, const float offsetY,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    const nn::util::Unorm8x4 Yellow ={ { 255, 255, 0, 255 } };

    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor(Yellow);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Save Log...");
}

void DrawOutputData(
    const float offsetX, const float offsetY,
    AttachmentOutput* pOutput,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };
    const int fontSize = 12;
    auto index = 0;

    pTextWriter->SetScale(0.8f, 0.8f);
    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[SixAxis] X:%8.5f(%0.2f[deg]) Y:%8.5f(%0.2f[deg]) Z:%8.5f(%0.2f[deg])",
        pOutput->sixAxisData.acceleration.x, pOutput->sixAxisDegreeData.GetX(),
        pOutput->sixAxisData.acceleration.y, pOutput->sixAxisDegreeData.GetY(),
        pOutput->sixAxisData.acceleration.z, pOutput->sixAxisDegreeData.GetZ());
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[OutMostFrame] Xmin:%d Ymin:%d Xmax:%d Ymax:%d", pOutput->outMostFrame.x, pOutput->outMostFrame.y,
        pOutput->outMostFrame.x + pOutput->outMostFrame.width, pOutput->outMostFrame.y + pOutput->outMostFrame.height);
    pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
    pTextWriter->Print("[CenterPoint] X:%0.1f Y:%0.1f", pOutput->centerPoint.x, pOutput->centerPoint.y);
    for (auto i = 0; i< pOutput->objectCount; i++)
    {
        pTextWriter->SetCursor(offsetX, offsetY + fontSize * index++);
        pTextWriter->Print("[Cluster %2d] Xmin:%03d Ymin:%03d Xmax:%03d Ymax:%03d",
            i,
            pOutput->clusters[i].measuredClusteringData.bound.x,
            pOutput->clusters[i].measuredClusteringData.bound.y,
            pOutput->clusters[i].measuredClusteringData.bound.x + pOutput->clusters[i].measuredClusteringData.bound.width,
            pOutput->clusters[i].measuredClusteringData.bound.y + pOutput->clusters[i].measuredClusteringData.bound.height);
    }
}

void CreateSampler(GraphicsSystem* pGraphicsSystem, Sampler* pSampler)
{
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->sampler.Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(&pSampler->descriptor, pSampler->sampler);
}

void CreateTexture(GraphicsSystem* pGraphicsSystem, int width, int height, nn::gfx::ImageFormat format, Texture* pTexture)
{
    {
        nn::gfx::Texture::InfoType textureInfo;

        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageFormat(format);

        pGraphicsSystem->AllocateTexture(&pTexture->texture, &textureInfo);
    }

    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(format);
        viewInfo.SetTexturePtr(&pTexture->texture);
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);

        pTexture->view.Initialize(&pGraphicsSystem->GetDevice(), viewInfo);
    }

    {
        pGraphicsSystem->RegisterTextureViewSlot(&pTexture->descriptor, pTexture->view);
    }
}

void ReflectionCheckerView::DrawBrightness(
    const float offsetX, const float offsetY,
    float averageBrightness,
    float maxBrightness,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 Black ={ { 0, 0, 0, 255 } };
    const nn::util::Unorm8x4 Red ={ { 255, 0, 0, 255 } };

    pTextWriter->SetScale(4.0f, 4.0f);

    if (m_ViewMode == ViewMode_Black)
    {
        pTextWriter->SetTextColor(Black);
    }
    else if (m_ViewMode == ViewMode_Red)
    {
        pTextWriter->SetTextColor(Red);
    }
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Brightness:\n");
    pTextWriter->Print("  (AVG) %0.3f \n  (MAX) %0.3f", averageBrightness, maxBrightness);
}

void ReflectionCheckerView::DrawStatus(
    const float offsetX, const float offsetY,
    nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4 Black ={ { 0, 0, 0, 255 } };

    pTextWriter->SetScale(4.0f, 4.0f);

    pTextWriter->SetTextColor(Black);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Controller Unconnected");
}

ReflectionCheckerView::ReflectionCheckerView()
    : m_NpadId(0)
    , m_MenuSelection(0)
    , m_RecordMenuIndex(RecordMenu_Camera)
    , m_pGraphicsSystem(nullptr)
    , m_pImageBuffer(nullptr)
    , m_PreviousPadState()
    , m_CurrentProcessor(IrSensorMode::IrSensorMode_Moment)
    , m_NextProcessor(IrSensorMode::IrSensorMode_Moment)
    , m_IrCameraHandle()
    , m_AttachmentLogger()
    , m_ViewMode(ViewMode_Black)
    , m_RecordInfo()
    , m_RecordFinishedCounter(0)
    , m_IsCaptureCalled(false)
    , m_IsCaptured(false)
{
    for (auto i = 0; i < IrSensorMode_Count; i++)
    {
        m_ModeStates[i] = nullptr;
    }

    // 時刻ライブラリ初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
}

ReflectionCheckerView::~ReflectionCheckerView()
{
    // 時刻ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
}

void ReflectionCheckerView::Initialize(nn::hid::NpadIdType id, GraphicsSystem* pGraphicsSystem, void* pImageTransferWorkMemory)
{
    m_NpadId = id;
    m_pGraphicsSystem = pGraphicsSystem;
    m_pImageBuffer = pImageTransferWorkMemory;

    // 6軸のハンドル取得
    int handleCount = nn::hid::GetSixAxisSensorHandles(&m_SixAxisHandles[0],
        nn::hid::NpadSixAxisSensorHandleCountMax,
        m_NpadId,
        nn::hid::NpadStyleJoyDual::Mask);
    NN_LOG("HandleCount(JoyDual)=%d\n", handleCount);
    nn::hid::StartSixAxisSensor(m_SixAxisHandles[1]);

    //IR カメラのハンドルの取得
    m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);

    //IR カメラの初期化
    nn::irsensor::Initialize(m_IrCameraHandle);
    NN_LOG("NpadPlayerNumber(%d)\n", m_NpadId);

    // IrSensor を Clustering として初期化
    m_ModeStates[IrSensorMode_Moment] = new MomentModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);
    m_ModeStates[IrSensorMode_ImageTransfer] = new ImageTransferModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle, pImageTransferWorkMemory, pGraphicsSystem);

    m_CurrentProcessor = m_NextProcessor = IrSensorMode_Moment;

    m_ModeStates[m_CurrentProcessor]->Start();
}

void ReflectionCheckerView::Finalize()
{
    // View の終了処理
    m_ModeStates[m_CurrentProcessor]->Stop();
    delete m_ModeStates[IrSensorMode_Moment];
    delete m_ModeStates[IrSensorMode_ImageTransfer];
    nn::irsensor::Finalize(m_IrCameraHandle);
}


void ReflectionCheckerView::Update() NN_NOEXCEPT
{
    if (!ProcessHidInput())
    {
        // return false;
        return;
    }

    m_IrStatus = nn::irsensor::GetIrCameraStatus(m_IrCameraHandle);

    m_ModeStates[m_CurrentProcessor]->Update();
}

void ReflectionCheckerView::Draw(GraphicsSystem* pGraphicsSystem, nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    if (m_IrStatus == nn::irsensor::IrCameraStatus_Available)
    {
        // モーメント情報の表示
        m_ModeStates[m_CurrentProcessor]->Render(&pGraphicsSystem->GetPrimitiveRenderer(), &pGraphicsSystem->GetCommandBuffer(), 0);

        // モーメント情報の取得
        float avgBrightness = 0.0f;
        float maxBrightness = 0.0f;
        if (m_CurrentProcessor == IrSensorMode_Moment)
        {
            m_ModeStates[m_CurrentProcessor]->GetBrightness(&avgBrightness, &maxBrightness);
        }

        DrawBrightness(250, 250, avgBrightness, maxBrightness, pTextWriter);
    }
    else if (m_IrStatus == nn::irsensor::IrCameraStatus_Unconnected)
    {
        DrawStatus(250, 300, pTextWriter);
    }
}

bool ReflectionCheckerView::IsFrameCaptured() NN_NOEXCEPT
{
    return m_IsCaptured;
}

void ReflectionCheckerView::CaptureFrame() NN_NOEXCEPT
{
    if (m_IsCaptureCalled)
    {
        m_IsCaptureCalled = false;
        m_pGraphicsSystem->CaptureFrame();
        m_IsCaptured = true;
    }
}

bool ReflectionCheckerView::ProcessHidInput() NN_NOEXCEPT
{
    nn::hid::NpadJoyDualState currentState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    nn::hid::GetNpadState(&currentState, m_NpadId);
    buttonDownSet = currentState.buttons & ~m_PreviousPadState.buttons;
    m_PreviousPadState = currentState;

    // Toggle ViewMode
    if ((buttonDownSet & nn::hid::NpadButton::R::Mask).IsAnyOn())
    {
        m_ViewMode = static_cast<ViewMode>((m_ViewMode + 1) % ViewMode_CountMax);

        m_ModeStates[m_CurrentProcessor]->SetViewMode(static_cast<IrSensorModeState::ViewMode>(m_ViewMode));
        m_ModeStates[m_CurrentProcessor]->Start();
    }

    // アプリ終了
    if ((buttonDownSet & nn::hid::NpadButton::Minus::Mask).IsAnyOn())
    {
        return false;
    }

    return true;
}//NOLINT(readability/fn_size)

int ReflectionCheckerView::GetMenuIndex() const
{
    const std::vector<MenuItem> menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
    int size = static_cast<int>(menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}
