﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "SceneBase.h"
#include "ReflectionCheckerScene.h"

#include <sstream>
namespace
{

const char* ToolVersion = "v0.3";

void DrawMenuToolState(const float offsetX, const float offsetY,
    GraphicsSystem* pGraphicsSystem)
{
    // Draw Menu Bar
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::CommandBuffer* pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // 背景を描画
    const float xStart = -1.0f;
    const float xEnd = 1.0f;
    const float yStart = 1.0f;
    float yEnd = -1.0f;
    //const float yEnd = 0.9f;

    nn::util::Vector3fType corner;
    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);

    nn::util::Vector3fType size;
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);


    const nn::util::Unorm8x4 Gray = { { 128, 128, 128, 255 } };
    pPrimitiveRenderer->SetColor(Gray);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size);

    yEnd = 0.9f;
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    const nn::util::Unorm8x4 Green = { { 0, 32, 0, 255 } };
    pPrimitiveRenderer->SetColor(Green);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size);

    // Write text
    nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();
    const nn::util::Unorm8x4 White ={ { 255, 255, 255, 255 } };

    pTextWriter->SetScale(1.2f, 1.2f);

    pTextWriter->SetTextColor(White);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Reflection Checker: %s", ToolVersion);
}

void DrawMenu(const float offsetX, const float offsetY,
    GraphicsSystem* pGraphicsSystem)
{
    NN_ASSERT_NOT_NULL(pGraphicsSystem);

    // 一段目メニュー
    DrawMenuToolState(offsetX, offsetY, pGraphicsSystem);
}

} // namespace


ReflectionCheckerScene::ReflectionCheckerScene()
    : m_Cursor(0)
    , m_View()
{
}

ReflectionCheckerScene::~ReflectionCheckerScene()
{
}

void ReflectionCheckerScene::DoInitialize()
{
    // View の初期化
    m_View.Initialize(m_NpadId, m_pGraphicsSystem, m_pImageBuffer);
    // キャプチャの開始
    m_pGraphicsSystem->InitializeFrameCapture();
}

void ReflectionCheckerScene::DoFinalize()
{
    // キャプチャの終了
    m_pGraphicsSystem->FinalizeFrameCapture();
    // 各 View の終了処理
    m_View.Finalize();
}


void ReflectionCheckerScene::DoUpdate() NN_NOEXCEPT
{
    if (!ProcessHidInput())
    {
        return;
    }

    m_View.Update();
}

void ReflectionCheckerScene::DoDraw() NN_NOEXCEPT
{
    DrawMenu(0.0f, 0.0f, m_pGraphicsSystem);
    m_View.Draw(m_pGraphicsSystem, &m_pGraphicsSystem->GetDebugFont());
}

bool ReflectionCheckerScene::ProcessHidInput() NN_NOEXCEPT
{
    nn::hid::NpadJoyDualState currentState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    nn::hid::GetNpadState(&currentState, m_NpadId);
    buttonDownSet = currentState.buttons & ~m_PreviousPadState.buttons;
    m_PreviousPadState = currentState;

    if ((buttonDownSet & nn::hid::NpadButton::Plus::Mask).IsAnyOn() || (buttonDownSet & nn::hid::NpadButton::Minus::Mask).IsAnyOn())
    {
        return false;
    }

    return true;

}

bool ReflectionCheckerScene::IsFrameCaptured()
{
    return m_View.IsFrameCaptured();
}

void ReflectionCheckerScene::CaptureFrame()
{
    m_View.CaptureFrame();
}
