﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
//#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/irsensor.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/init/init_Malloc.h>

#include "GraphicsSystem.h"
#include "ImageTransferModeState.h"
#include "AttachmentDataFormat.h"
#include "AttachmentEvaluationScene.h"
#include "SceneBase.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

namespace {

const int FrameRate = 60;

}

extern "C" void nnMain()
{
    /////////////////////////////////////////////////////////////////
    // Initialization
    /////////////////////////////////////////////////////////////////

    nn::hid::NpadIdType npadIds[] =
    {
        ::nn::hid::NpadId::No1,
        ::nn::hid::NpadId::No2
    };

    //
    // Memory
    //

    nn::mem::StandardAllocator appAllocator;
    void* workMemoryBuffers[NN_ARRAY_SIZE(npadIds)];
    const size_t appMemorySize  = 128 * 1024 * 1014;
    nn::Bit8* pAppMemory        = new nn::Bit8[appMemorySize];

    appAllocator.Initialize(pAppMemory, appMemorySize);


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    const size_t graphicsMemorySize = 8 * 1024 * 1024;
    void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
    nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif

    //
    // Graphics
    //

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&appAllocator);
    pGraphicsSystem->Initialize();

    //
    // Initialize Hid
    //
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(::nn::hid::NpadStyleJoyDual::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));

    //
    // Parse input file
    //



    //
    // Initialize Scene
    //
    const int SceneMaxCount = 1;
    AttachmentEvaluationScene evalScene[SceneMaxCount];

    enum AttachementId {
        Attachment_Rbt,
        Attachment_Pno,
        Attachment_Hus,
        Attachment_Calibration,
        Attachment_CountMax,
    };

    AttachmentInput inputList[Attachment_CountMax];

    // 入力 (将来的にはファイル読込)
    AttachmentInput input = {};
    std::strncpy(input.attachmentName, "Rbt", 32);
    input.safetyFramePixelX = 25;
    input.safetyFramePixelY = 22;

    // 期待される入力
    input.objectCount = 8;
    nn::irsensor::ClusteringData idealValue[nn::irsensor::ClusteringProcessorObjectCountMax] = {};
    idealValue[0].bound = { 27, 32, 19, 6 };               // 左上
    idealValue[1].bound = { (320 - 27 - 19), 32, 19, 6 };  // 右上
    idealValue[2].bound = { 25, 82, 19, 7 };               // 左真ん中上
    idealValue[3].bound = { (320 - 25 - 19), 82, 19, 7 };  // 右真ん中上
    idealValue[4].bound = { 25, (240 - 82 - 7), 19, 7 };              // 左真ん中下
    idealValue[5].bound = { (320 - 25 - 19), (240 - 82 - 7), 19, 7 }; // 右真ん中下
    idealValue[6].bound = { 27, (240 - 32 - 6), 19, 6 };              // 左下
    idealValue[7].bound = { (320 - 27 - 19), (240 - 32 - 6), 19, 6 }; // 右下

    for (auto i = 0; i < input.objectCount; i++)
    {
        input.expectedClusters[i].clusterId = i;
        input.expectedClusters[i].idealClusteringData = idealValue[i];
    }

    nn::irsensor::ClusteringProcessorConfig config;
    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);
    config.irCameraConfig.exposureTime = nn::TimeSpanType::FromMicroSeconds(200);
    config.irCameraConfig.gain = 4;
    config.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_NearObjects;
    config.objectIntensityMin = 150;
    config.objectPixelCountMin = 3;
    config.objectPixelCountMax = 65535;
    input.clusteringConfig = config;

    inputList[Attachment_Rbt] = input;

    // 入力 (将来的にはファイル読込)
    std::memset(&input, 0, sizeof(input));
    std::strncpy(input.attachmentName, "Pno", 32);
    input.safetyFramePixelX = 25;
    input.safetyFramePixelY = 25;

    // 期待される入力
    input.objectCount = 6;
    idealValue[0].bound = { 25, 198, 40, 15 };  // ダイヤル
    idealValue[1].bound = { 280, 180, 15, 8 };  // 読み込みボタン
    idealValue[2].bound = { 61, 59, 10, 30 };   // 基準マーカー(左)
    idealValue[3].bound = { 254, 59, 10, 30 };  // 基準マーカー(右)
    idealValue[4].bound = { 0, 125, 35, 10 };   // オクターブシフト(下)
    idealValue[5].bound = { 0, 60, 35, 10 };    // オクターブシフト(上)

    for (auto i = 0; i < input.objectCount; i++)
    {
        input.expectedClusters[i].clusterId = i;
        input.expectedClusters[i].idealClusteringData = idealValue[i];
    }

    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);
    config.irCameraConfig.exposureTime = nn::TimeSpanType::FromMicroSeconds(175);
    config.irCameraConfig.gain = 3;
    config.irCameraConfig.isNegativeImageUsed = false;
    config.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_NearObjects;
    config.objectIntensityMin = 150;
    config.isExternalLightFilterEnabled = false;
    config.objectPixelCountMin = 8;
    config.objectPixelCountMax = 76800;
    input.clusteringConfig = config;

    inputList[Attachment_Pno] = input;

    // 入力 (将来的にはファイル読込)
    std::memset(&input, 0, sizeof(input));
    std::strncpy(input.attachmentName, "Hus", 32);
    input.safetyFramePixelX = 25;
    input.safetyFramePixelY = 25;

    // 期待される入力
    input.objectCount = 2;
    idealValue[0].bound = { 50, 15, 20, 10 };  // 基準マーカ(左)
    idealValue[1].bound = { 170, 15, 20, 10 };  // 基準マーカ(右)

    for (auto i = 0; i < input.objectCount; i++)
    {
        input.expectedClusters[i].clusterId = i;
        input.expectedClusters[i].idealClusteringData = idealValue[i];
    }

    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);
    config.irCameraConfig.exposureTime = nn::TimeSpanType::FromMicroSeconds(32);
    config.irCameraConfig.gain = 16;
    config.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_NearObjects;
    config.windowOfInterest.x = 40;
    config.windowOfInterest.y = 60;
    config.windowOfInterest.width = 240;
    config.windowOfInterest.height = 120;
    config.objectIntensityMin = 140;
    config.objectPixelCountMin = 32;
    config.objectPixelCountMax = 4096;
    config.isExternalLightFilterEnabled = false;
    input.clusteringConfig = config;

    inputList[Attachment_Hus] = input;

    // 入力 (将来的にはファイル読込)
    std::memset(&input, 0, sizeof(input));
    std::strncpy(input.attachmentName, "Cal", 32);
    input.safetyFramePixelX = 0;
    input.safetyFramePixelY = 0;

    // 期待される入力
    input.objectCount = 9;
    idealValue[0].bound = { 25, 25, 5, 5 };   // 左上
    idealValue[1].bound = { 158, 22, 5, 5 };  // 真ん中上
    idealValue[2].bound = { 290, 25, 5, 5 };  // 右上
    idealValue[3].bound = { 22, 118, 5, 5 };  // 左中
    idealValue[4].bound = { 145, 105, 30, 30 };  // 中
    idealValue[5].bound = { 293, 118, 5, 5 };  // 右中
    idealValue[6].bound = { 25, 210, 5, 5 };  // 左下
    idealValue[7].bound = { 158, 213, 5, 5 };  // 真ん中下
    idealValue[8].bound = { 290, 210, 5, 5 };  // 右下

    for (auto i = 0; i < input.objectCount; i++)
    {
        input.expectedClusters[i].clusterId = i;
        input.expectedClusters[i].idealClusteringData = idealValue[i];
    }

    nn::irsensor::GetClusteringProcessorDefaultConfig(&config);
    config.irCameraConfig.exposureTime = nn::TimeSpanType::FromMicroSeconds(150);
    config.irCameraConfig.gain = 1;
    config.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_AllObjects;
    config.objectIntensityMin = 200;
    config.objectPixelCountMin = 3;
    config.objectPixelCountMax = 4096;
    config.isExternalLightFilterEnabled = true;
    input.clusteringConfig = config;

    inputList[Attachment_Calibration] = input;

    for (int i = 0; i < SceneMaxCount; i++)
    {
        workMemoryBuffers[i] = appAllocator.Allocate(ImageTransferModeState::WorkMemorySize, ImageTransferModeState::WorkMemoryAlignement);
        SceneBase* pScene = &evalScene[i];
        pScene->Initialize(npadIds[i], pGraphicsSystem, workMemoryBuffers[i], Attachment_CountMax, &inputList[0]);
    }

    /////////////////////////////////////////////////////////////////
    // Main Loop
    /////////////////////////////////////////////////////////////////

    bool exit = false;
    while (!exit)
    {
        // Update
        for (auto i = 0; i < SceneMaxCount; i++)
        {
            SceneBase* pScene = &evalScene[i];
            pScene->Update();
            //if (false)
            //{
            //    exit = true;
            //}
        }

        // Draw
        nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();
        pGraphicsSystem->BeginDraw();

        for (auto i = 0; i < SceneMaxCount; ++i)
        {
            SceneBase* pScene = &evalScene[i];
            pScene->Draw();
        }
        pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());

        // キャプチャする
        for (auto i = 0; i < SceneMaxCount; ++i)
        {
            evalScene[i].CaptureFrame();
        }

        pGraphicsSystem->EndDraw();

        bool isFrameCaptured = false;
        for (auto i = 0; i < SceneMaxCount; ++i)
        {
            if(evalScene[i].IsFrameCaptured())
            {
                isFrameCaptured = true;
            }
        }

        if (isFrameCaptured)
        {
            pGraphicsSystem->Synchronize(
                nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000));
        }
        else
        {
            pGraphicsSystem->Synchronize(
                nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
        }
    }
    /////////////////////////////////////////////////////////////////
    // Finalization
    /////////////////////////////////////////////////////////////////

    //
    // Finalize Demo
    //
    for (int i = 0; i < SceneMaxCount; ++i)
    {
        SceneBase* pScene = &evalScene[i];
        pScene->Finalize();
        appAllocator.Free(workMemoryBuffers[i]);
    }

    //
    // Finalize Graphics
    //

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    //
    // Finalize Memory
    //
    appAllocator.Finalize();
    delete[] pAppMemory;
}//NOLINT(readability/fn_size)
