﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ClusteringModeState.h"

#include <algorithm>
#include <sstream>
#include <cstring>

#include <nn/nn_Assert.h>

namespace
{

void WriteClusteringProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::ClusteringProcessorConfig& clusteringConfig,
    const nn::irsensor::ClusteringProcessorState* pClusteringState,
    const AttachmentOutput& output,
    const int screenIndex) NN_NOEXCEPT
{
    NN_UNUSED(screenIndex);

    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pClusteringState);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green ={ { 0, 255, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    // 比が3:2 になるように計算
    const float xStart = -0.95f, xEnd = 0.0f;
    const float yStart = 0.72f, yEnd = -0.4f;

    pPrimitiveRenderer->SetLineWidth(3.0f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 各面を描画
    for (auto i = 0; i < pClusteringState->objectCount; ++i)
    {
        const nn::irsensor::ClusteringData* pObject = &pClusteringState->objects[i];
        float boundStartX = static_cast<float>(pObject->bound.x + clusteringConfig.windowOfInterest.x);
        float boundStartY = static_cast<float>(pObject->bound.y + clusteringConfig.windowOfInterest.y);
        float boundEndX = static_cast<float>(pObject->bound.x + pObject->bound.width + clusteringConfig.windowOfInterest.x);
        float boundEndY = static_cast<float>(pObject->bound.y + pObject->bound.height + clusteringConfig.windowOfInterest.y);

        pPrimitiveRenderer->SetColor(Green);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundEndX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundEndY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * boundStartX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * boundStartY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }

    // 稼動フレームの表示
    const nn::util::Unorm8x4 Cyan = { { 128, 0, 128, 255 } };
    pPrimitiveRenderer->SetColor(Cyan);

    int boundX = output.outMostFrame.x + clusteringConfig.windowOfInterest.x;
    int boundY = output.outMostFrame.y + clusteringConfig.windowOfInterest.y;
    int boundW = output.outMostFrame.width;
    int boundH = output.outMostFrame.height;

    if (boundX + boundW > clusteringConfig.windowOfInterest.x + clusteringConfig.windowOfInterest.width)
    {
        boundW = clusteringConfig.windowOfInterest.x + clusteringConfig.windowOfInterest.width - boundX;
    }
    if (boundY + boundH > clusteringConfig.windowOfInterest.y + clusteringConfig.windowOfInterest.height)
    {
        boundH = clusteringConfig.windowOfInterest.y + clusteringConfig.windowOfInterest.height - boundY;
    }

    nn::util::VectorSet(&begin,
        xStart + (xEnd - xStart) * boundX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
        yStart + (yEnd - yStart) * boundY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    nn::util::VectorSet(&end,
        xStart + (xEnd - xStart) * (boundX + boundW - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
        yStart + (yEnd - yStart) * boundY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin,
        xStart + (xEnd - xStart) * (boundX + boundW - 1) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
        yStart + (yEnd - yStart) * (boundY + boundH - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end,
        xStart + (xEnd - xStart) * boundX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
        yStart + (yEnd - yStart) * (boundY + boundH - 1) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin,
        xStart + (xEnd - xStart) * boundX / static_cast<float>(nn::irsensor::IrCameraImageWidth),
        yStart + (yEnd - yStart) * boundY / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
}

void UpdateOutmostWindow(AttachmentOutput* pAttachmentOutput, const nn::irsensor::ClusteringProcessorState& state)
{
    NN_ASSERT_NOT_NULL(pAttachmentOutput);

    for (auto i = 0; i < state.objectCount; i++)
    {
        int16_t leftMost = state.objects[i].bound.x;
        int16_t rightMost = state.objects[i].bound.x + state.objects[i].bound.width - 1;
        int16_t upMost = state.objects[i].bound.y;
        int16_t downMost = state.objects[i].bound.y + state.objects[i].bound.height - 1;

        if (pAttachmentOutput->outMostFrame.x > leftMost)
        {
            pAttachmentOutput->outMostFrame.width += (pAttachmentOutput->outMostFrame.x - leftMost);
            pAttachmentOutput->outMostFrame.x = leftMost;
        }
        if (pAttachmentOutput->outMostFrame.x + pAttachmentOutput->outMostFrame.width - 1 < rightMost)
        {
            pAttachmentOutput->outMostFrame.width = rightMost - pAttachmentOutput->outMostFrame.x + 1;
        }
        if (pAttachmentOutput->outMostFrame.y > upMost)
        {
            pAttachmentOutput->outMostFrame.height += (pAttachmentOutput->outMostFrame.y - upMost);
            pAttachmentOutput->outMostFrame.y = upMost;
        }
        if (pAttachmentOutput->outMostFrame.y + pAttachmentOutput->outMostFrame.height - 1 < downMost)
        {
            pAttachmentOutput->outMostFrame.height = downMost - pAttachmentOutput->outMostFrame.y + 1;
        }
    }
    // 中心座標の計算
    pAttachmentOutput->centerPoint.x = pAttachmentOutput->outMostFrame.x + pAttachmentOutput->outMostFrame.width / 2.0f;
    pAttachmentOutput->centerPoint.y = pAttachmentOutput->outMostFrame.y + pAttachmentOutput->outMostFrame.height / 2.0f;
}

void MatchClusters(AttachmentInput* pInput, AttachmentOutput* pAttachmentOutput, const nn::irsensor::ClusteringProcessorState& state)
{
    NN_ASSERT_NOT_NULL(pInput);
    NN_ASSERT_NOT_NULL(pAttachmentOutput);

    for (auto i = 0; i < state.objectCount; i++)
    {
        float centerX = state.objects[i].centroid.x;
        float centerY = state.objects[i].centroid.y;
        int inputMinX = state.objects[i].bound.x;
        int inputMaxX = state.objects[i].bound.x + state.objects[i].bound.width - 1;
        int inputMinY = state.objects[i].bound.y;
        int inputMaxY = state.objects[i].bound.y + state.objects[i].bound.height - 1;
        float inputDistance = 0.0f;

        for (auto j = 0 ; j < pInput->objectCount; j++)
        {
            // X軸方向には大きくとりたいので 25px マージン
            int minX = pInput->expectedClusters[j].idealClusteringData.bound.x - IrCameraNeighborMarginPixel;
            int maxX = pInput->expectedClusters[j].idealClusteringData.bound.x + pInput->expectedClusters[j].idealClusteringData.bound.width - 1 + IrCameraNeighborMarginPixel;
            // Y軸方向には他と領域がかぶらないように 15px マージン
            int minY = pInput->expectedClusters[j].idealClusteringData.bound.y - IrCameraSafetyFramePixel;
            int maxY = pInput->expectedClusters[j].idealClusteringData.bound.y + pInput->expectedClusters[j].idealClusteringData.bound.height - 1 + IrCameraSafetyFramePixel;

            if (centerX > minX && centerX < maxX && centerY > minY && centerY < maxY)
            {
                // すでに登録済みの場合は既存のクラスタと比較
                if (pAttachmentOutput->clusters[j].isValid)
                {
                    int halfWidth = pInput->clusteringConfig.windowOfInterest.width / 2;
                    int halfHeight = pInput->clusteringConfig.windowOfInterest.height / 2;

                    int outputMinX = 0;
                    int outputMaxX = pInput->clusteringConfig.windowOfInterest.width;
                    int outputMinY = 0;
                    int outputMaxY = pInput->clusteringConfig.windowOfInterest.height;
                    float outputDistance = 0.0f;
                    // 期待値クラスタの象限を判定して、画角の外側方向へのみ更新するようにする
                    if ((minX + maxX) / 2 > halfWidth)
                    {
                        if ((minY + maxY) / 2 < halfHeight)
                        {
                            // 第一象限
                            outputMaxX = pAttachmentOutput->clusters[j].measuredClusteringData.bound.x + pAttachmentOutput->clusters[j].measuredClusteringData.bound.width - 1;
                            outputMinY = pAttachmentOutput->clusters[j].measuredClusteringData.bound.y;
                            outputDistance = sqrt(powf(static_cast<float>(outputMaxX - halfWidth), 2.f)
                                + powf(static_cast<float>(outputMinY - halfHeight), 2.f));
                            inputDistance = sqrt(powf(static_cast<float>(inputMaxX - halfWidth), 2.f)
                                + powf(static_cast<float>(inputMinY - halfHeight), 2.f));
                        }
                        else
                        {
                            // 第四象限
                            outputMaxX = pAttachmentOutput->clusters[j].measuredClusteringData.bound.x + pAttachmentOutput->clusters[j].measuredClusteringData.bound.width - 1;
                            outputMaxY = pAttachmentOutput->clusters[j].measuredClusteringData.bound.y + pAttachmentOutput->clusters[j].measuredClusteringData.bound.height - 1;
                            outputDistance = sqrt(powf(static_cast<float>(outputMaxX - halfWidth), 2.f)
                                + powf(static_cast<float>(outputMaxY - halfHeight), 2.f));
                            inputDistance = sqrt(powf(static_cast<float>(inputMaxX - halfWidth), 2.f)
                                + powf(static_cast<float>(inputMaxY - halfHeight), 2.f));
                        }
                    }
                    else
                    {
                        if ((minY + maxY) / 2 < halfHeight)
                        {
                            // 第二象限
                            outputMinX = pAttachmentOutput->clusters[j].measuredClusteringData.bound.x;
                            outputMinY = pAttachmentOutput->clusters[j].measuredClusteringData.bound.y;
                            outputDistance = sqrt(powf(static_cast<float>(outputMinX - halfWidth), 2.f)
                                + powf(static_cast<float>(outputMinY - halfHeight), 2.f));
                            inputDistance = sqrt(powf(static_cast<float>(inputMinX - halfWidth), 2.f)
                                + powf(static_cast<float>(inputMinY - halfHeight), 2.f));
                        }
                        else
                        {
                            // 第三象限
                            outputMinX = pAttachmentOutput->clusters[j].measuredClusteringData.bound.x;
                            outputMaxY = pAttachmentOutput->clusters[j].measuredClusteringData.bound.y + pAttachmentOutput->clusters[j].measuredClusteringData.bound.height - 1;
                            outputDistance = sqrt(powf(static_cast<float>(outputMinX - halfWidth), 2.f)
                                + powf(static_cast<float>(outputMaxY - halfHeight), 2.f));
                            inputDistance = sqrt(powf(static_cast<float>(inputMinX - halfWidth), 2.f)
                                + powf(static_cast<float>(inputMaxY - halfHeight), 2.f));
                        }
                    }

                    // より外側だった場合は更新
                    if (outputDistance < inputDistance)
                    {
                        pAttachmentOutput->clusters[j].measuredClusteringData = state.objects[i];
                    }
                }
                else
                {
                    // 初回は期待値の枠 + カメラ安全フレームに入っていれば、対象のクラスタと判定して更新する
                    pAttachmentOutput->clusters[j].clusterId = j;
                    pAttachmentOutput->clusters[j].isValid = true;
                    pAttachmentOutput->clusters[j].measuredClusteringData = state.objects[i];
                    pAttachmentOutput->objectCount++;
                }
            }
        }
    }
}

void UpdateClusteringData(AttachmentInput* pInput, AttachmentOutput* pAttachmentOutput, const nn::irsensor::ClusteringProcessorState& state)
{
    NN_ASSERT_NOT_NULL(pAttachmentOutput);
    // 最外矩形の更新
    UpdateOutmostWindow(pAttachmentOutput, state);

    // インデックスごとのマッチング
    MatchClusters(pInput, pAttachmentOutput, state);

    // 生データはそのまま保存
    pAttachmentOutput->objectRawCount = state.objectCount;
    for (auto i = 0; i < state.objectCount; i++)
    {
        pAttachmentOutput->rawClusterData[i] = state.objects[i];
    }
}

} // namespace

ClusteringModeState::ClusteringModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle,
    AttachmentInput* pInput, AttachmentOutput* pOutput, AttachmentLogger* pLogger)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
{
    m_pInput = pInput;
    m_pOutput = pOutput;
    m_pLogger = pLogger;

    nn::irsensor::GetClusteringProcessorDefaultConfig(&m_ClusteringProcessorConfig);

    // Input から入力を得る
    m_ClusteringProcessorConfig = pInput->clusteringConfig;

    nn::irsensor::ClusteringProcessorConfig* pClusteringConfig = &m_ClusteringProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pClusteringConfig->irCameraConfig,
        nn::irsensor::ClusteringProcessorExposureTimeMin,
        nn::irsensor::ClusteringProcessorExposureTimeMax
    );

    m_ReadWriteMenu.emplace_back("Object Pixel Count Min",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectPixelCountMin;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectPixelCountMin += delta * 16;
            pClusteringConfig->objectPixelCountMin =
            std::min(std::max(pClusteringConfig->objectPixelCountMin, 0), pClusteringConfig->objectPixelCountMax - 1);
        }
    );
    m_ReadWriteMenu.emplace_back("Object Pixel Count Max",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectPixelCountMax;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectPixelCountMax += delta * 16;
            pClusteringConfig->objectPixelCountMax =
            std::max(std::min(pClusteringConfig->objectPixelCountMax, nn::irsensor::ClusteringProcessorObjectPixelCountMax), pClusteringConfig->objectPixelCountMin + 1);
        }
    );
    m_ReadWriteMenu.emplace_back("Object Intensity Min",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectIntensityMin;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectIntensityMin += delta * 8;
            pClusteringConfig->objectIntensityMin =
            std::max(std::min(pClusteringConfig->objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
        }
    );
    m_ReadWriteMenu.emplace_back("External Light Filter",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << (pClusteringConfig->isExternalLightFilterEnabled ? "Enabled" : "Disabled");
        },
        [pClusteringConfig](int8_t delta) {
            NN_UNUSED(delta);
            pClusteringConfig->isExternalLightFilterEnabled = !pClusteringConfig->isExternalLightFilterEnabled;
        }
    );

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_ClusteringProcessorState.samplingNumber, &m_ClusteringProcessorState.ambientNoiseLevel);
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);
    int8_t* pObjectCount = &m_ClusteringProcessorState.objectCount;
    m_ReadOnlyMenu.emplace_back("Object Count",
        [pObjectCount](std::stringstream& sstr) {
            sstr << static_cast<int>(*pObjectCount);
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    // 出力データ初期化
    Reset();
}

ClusteringModeState::~ClusteringModeState()
{
}

void ClusteringModeState::Start()
{
    nn::irsensor::RunClusteringProcessor(m_IrCameraHandle, m_ClusteringProcessorConfig);
}

void ClusteringModeState::Update()
{
    int count;
    nn::Result result = nn::irsensor::GetClusteringProcessorStates(&m_ClusteringProcessorState, &count, 1, m_IrCameraHandle);
    HandleResult(result);
    m_ClusteringProcessorState.objectCount = (m_ClusteringProcessorState.objectCount >= nn::irsensor::ClusteringProcessorObjectCountMax) ? 0 : m_ClusteringProcessorState.objectCount;
    m_Statistics.Update(&m_ClusteringProcessorState.samplingNumber, IrSensorModeStatistics::ExpectedClusteringModeFramerate);

    // 出力バッファの更新
    UpdateClusteringData(m_pInput, m_pOutput, m_ClusteringProcessorState);
}

void ClusteringModeState::Record(const RecordInfo& recordInfo, const PathInfo& pathInfo, int clusterCountMax, GraphicsSystem* pGraphicsSystem)
{
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    m_pLogger->Initialize(pathInfo.directoryName);
    m_pLogger->AppendLineFormat(clusterCountMax);
    m_pLogger->AppendLine(*m_pOutput, recordInfo, clusterCountMax);
    m_pLogger->Write(pathInfo.logFileName);
    m_pLogger->WritePng(pathInfo.pngFileName, pGraphicsSystem);
    m_pLogger->Finalize();
#else
    NN_UNUSED(recordInfo);
    NN_UNUSED(pathInfo);
    NN_UNUSED(clusterCountMax);
    NN_UNUSED(pGraphicsSystem);
#endif
}

void ClusteringModeState::Reset()
{
    // Output データのクリア
    m_pOutput->outMostFrame.x = m_pInput->clusteringConfig.windowOfInterest.width;
    m_pOutput->outMostFrame.y = m_pInput->clusteringConfig.windowOfInterest.height;
    m_pOutput->outMostFrame.width = -m_pInput->clusteringConfig.windowOfInterest.width;
    m_pOutput->outMostFrame.height = -m_pInput->clusteringConfig.windowOfInterest.height;

    m_pOutput->objectCount = 0;
    for (auto i = 0; i < nn::irsensor::ClusteringProcessorObjectCountMax; i++)
    {
        m_pOutput->clusters[i].clusterId = 0;
        m_pOutput->clusters[i].isValid = false;
        m_pOutput->clusters[i].measuredClusteringData.averageIntensity = 0.0f;
        m_pOutput->clusters[i].measuredClusteringData.centroid.x = 0.0f;
        m_pOutput->clusters[i].measuredClusteringData.centroid.y = 0.0f;
        m_pOutput->clusters[i].measuredClusteringData.pixelCount = 0;
        m_pOutput->clusters[i].measuredClusteringData.bound.x = 0;
        m_pOutput->clusters[i].measuredClusteringData.bound.y = 0;
        m_pOutput->clusters[i].measuredClusteringData.bound.width = 0;
        m_pOutput->clusters[i].measuredClusteringData.bound.height = 0;
    }

    m_pOutput->objectRawCount = 0;
    for (auto i = 0; i < nn::irsensor::ClusteringProcessorObjectCountMax; i++)
    {
        m_pOutput->rawClusterData[i].averageIntensity = 0.0f;
        m_pOutput->rawClusterData[i].centroid.x = 0.0f;
        m_pOutput->rawClusterData[i].centroid.y = 0.0f;
        m_pOutput->rawClusterData[i].pixelCount = 0;
        m_pOutput->rawClusterData[i].bound.x = 0;
        m_pOutput->rawClusterData[i].bound.y = 0;
        m_pOutput->rawClusterData[i].bound.width = 0;
        m_pOutput->rawClusterData[i].bound.height= 0;
    }

    memset(&m_pOutput->sixAxisData, 0, sizeof(nn::hid::SixAxisSensorState));

}

void ClusteringModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WriteClusteringProcessorState(pPrimitiveRenderer, pCommandBuffer, m_ClusteringProcessorConfig, &m_ClusteringProcessorState, *m_pOutput, index);
}
