﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <cmath>

#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_ScopeExit.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nnt.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/audio.h>

namespace {

const float Vertices[] =
{
    0.0f, 0.5f, 0.0f, 0.0f, 0.0f,
    -0.5f, 0.0f, 0.0f, 0.0f, 1.0f,
    0.5f, 0.0f, 0.0f, 1.0f, 0.0f,
    0.0f, -0.5f, 0.0f, 1.0f, 1.0f
};

const int Indices[] =
{
    0, 1, 2, 3
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

nn::hid::NpadHandheldState g_NpadHandheldState;

//-----------------------------------------------------------------------------
// メモリ

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
const int RenderWidth = 1280;
const int RenderHeight = 720;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    16 * 1024 * 1024,
    20 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

size_t g_MaxScratchMemory = 0;
ptrdiff_t g_ScratchMemoryOffset = 0;

//-----------------------------------------------------------------------------
// グラフィックス

// リソースの読み込み
void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    g_pMemory.AlignUp( alignment );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

// レイヤを初期化
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

// デバイスを初期化
nn::gfx::Device g_Device;
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

// メモリプールを初期化
nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( &g_Device, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderWidth );
    info.SetHeight( RenderHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( 2 );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( &g_Device, info ) );
        g_SwapChain.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( &g_Device, info );
}

// コマンドバッファを初期化
nn::gfx::CommandBuffer g_CommandBuffer;
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize( &g_Device, info );
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderWidth );
        scissorInfo.SetHeight( RenderHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( &g_Device, info );
}

// ラスタライザを初期化
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( &g_Device, info );
}

// BlendState を初期化
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( &g_Device, info );
}

// DepthStencilState を初期化
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    g_DepthStencilState.Initialize( &g_Device, info );
}

// VertexState を初期化
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    uint16_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
        attribs[ 0 ].SetShaderSlot( 0 );
    }
    {
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
        attribs[ 1 ].SetShaderSlot( 1 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( &g_Device, info, NULL );
}

// 頂点用のバッファを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_VertexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデックス用のバッファを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_IndexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 定数バッファ用のバッファを初期化
nn::gfx::Buffer g_ConstantBuffer;
nn::gfx::Buffer g_MatrixConstantBuffer;
void InitializeConstantBuffer()
{
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(sizeof(float) * 4);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
        {
            ptrdiff_t& offset = g_MemoryPoolOffset[MemoryPoolType_CpuUncached_GpuCached];
            offset = nn::util::align_up(offset, nn::gfx::Buffer::GetBufferAlignment(&g_Device, info));
            g_ConstantBuffer.Initialize(&g_Device, info,
                &g_MemoryPool[MemoryPoolType_CpuUncached_GpuCached], offset, info.GetSize());
            offset += info.GetSize();
        }
        else
        {
            g_ConstantBuffer.Initialize(&g_Device, info, NULL, 0, 0);
        }
    }

    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(sizeof(float) * 16);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
        {
            ptrdiff_t& offset = g_MemoryPoolOffset[MemoryPoolType_CpuUncached_GpuCached];
            offset = nn::util::align_up(offset, nn::gfx::Buffer::GetBufferAlignment(&g_Device, info));
            g_MatrixConstantBuffer.Initialize(&g_Device, info,
                &g_MemoryPool[MemoryPoolType_CpuUncached_GpuCached], offset, info.GetSize());
            offset += info.GetSize();
        }
        else
        {
            g_MatrixConstantBuffer.Initialize(&g_Device, info, NULL, 0, 0);
        }
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( &g_Device, info );
}

// 各ディスクリプタプールを初期化
nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 2 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
    g_BufferDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_SamplerDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_TextureDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

// リソースの読み込み
nn::gfx::ResTextureFile* g_pResTextureFile;
void InitializeResTextureFile()
{
    void *pResource = ReadResource( "Contents:/SampleTexture.bntx" );
    g_pResTextureFile = nn::gfx::ResTextureFile::ResCast( pResource );
    g_pResTextureFile->Initialize( &g_Device );
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Initialize( &g_Device );
    }
}

void FinalizeResTextureFile()
{
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Finalize( &g_Device );
    }
    g_pResTextureFile->Finalize( &g_Device );
}

nn::gfx::ResShaderFile* g_pResShaderFile;
void InitializeResShaderFile()
{
    void* pResource = ReadResource( "Contents:/SampleShader.bnsh" );
    g_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pResource );
    nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( &g_Device );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        nn::gfx::ShaderInitializeResult result = pProgram->Initialize( &g_Device );
        NN_ASSERT_EQUAL( result, nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

#if NN_GFX_IS_TARGET_NVN
    g_MaxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize( &g_Device, &g_pResShaderFile, 1 );
    int scratchMemoryAlignment;
    nvnDeviceGetInteger( g_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &scratchMemoryAlignment );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
    offset = nn::util::align_up( offset, scratchMemoryAlignment );
    g_ScratchMemoryOffset = offset;
    offset += g_MaxScratchMemory;
#endif
    NN_UNUSED( g_MaxScratchMemory );
    NN_UNUSED( g_ScratchMemoryOffset );
}

// リソースの開放
void FinalizeResShaderFile()
{
    auto pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        pProgram->Finalize( &g_Device );
    }

    pContainer->Finalize( &g_Device );
}

// 各オブジェクトを初期化
void InitializeGfxObjects()
{
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( g_Device );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeConstantBuffer();
    InitializeSampler();

    InitializeBufferDescriptorPool();
    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeResTextureFile();
    InitializeResShaderFile();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

// グラフィックスの更新
nn::util::Vector3f g_Position(0, 0, 0);
void UpdateGfx()
{
    g_Position.Set(
        static_cast<float>(g_NpadHandheldState.analogStickL.x) / nn::hid::AnalogStickMax,
        static_cast<float>(g_NpadHandheldState.analogStickL.y) / nn::hid::AnalogStickMax,
        0);
}

// コマンドを生成
void MakeCommand( int frame )
{
    nn::gfx::GpuAddress gpuAddress;

    g_CommandBuffer.Reset();
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    // ワンタイムのコマンドバッファには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリプールを使うことがより適しています
    g_CommandBuffer.AddCommandMemory(
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 * 1024 );
    g_pMemory.AlignUp( 256 );
    g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
    g_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();

        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        g_CommandBuffer.SetDescriptorPool( &g_BufferDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );

        g_CommandBuffer.ClearColor( pTarget, 0.3f, 0.1f, 0.1f, 1.0f, NULL );

#if NN_GFX_IS_TARGET_NVN
        nvnCommandBufferSetShaderScratchMemory( g_CommandBuffer.ToData()->pNvnCommandBuffer,
            g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ].ToData()->pNvnMemoryPool,
            g_ScratchMemoryOffset, g_MaxScratchMemory );
#endif

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        int variation = frame / 32 % pContainer->GetShaderVariationCount();
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( variation );
        nn::gfx::Shader* pShader = pProgram->GetShader();
        g_CommandBuffer.SetShader( pShader, nn::gfx::ShaderStageBit_All );

        g_CommandBuffer.SetRenderTargets( 1, &pTarget, NULL );

        g_CommandBuffer.SetViewportScissorState( &g_ViewportScissor );

        g_CommandBuffer.SetRasterizerState( &g_RasterizerState );
        g_CommandBuffer.SetBlendState( &g_BlendState );
        g_CommandBuffer.SetDepthStencilState( &g_DepthStencilState );
        g_CommandBuffer.SetVertexState( &g_VertexState );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

        nn::gfx::DescriptorSlot matrixConstantBufferDescriptor;
        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot( &matrixConstantBufferDescriptor, g_BufferDescriptorBaseIndex + 1 );
        g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor, g_BufferDescriptorBaseIndex );
        g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
        g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );

        int slotMatrix = pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Matrix");
        int slotMat = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        int slotTex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
        if( slotMatrix >= 0 )
        {
            g_CommandBuffer.SetConstantBuffer( slotMatrix,
                nn::gfx::ShaderStage_Vertex, matrixConstantBufferDescriptor );
        }
        if( slotMat >= 0 )
        {
            g_CommandBuffer.SetConstantBuffer( slotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
        }
        if( slotTex >= 0 )
        {
            g_CommandBuffer.SetTextureAndSampler( slotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip,
            nn::gfx::IndexFormat_Uint32, gpuAddress, 4, 0 );
    }
    g_CommandBuffer.End();
}

// 各オブジェクトを破棄
void FinalizeGfxObjects()
{
    FinalizeResShaderFile();
    FinalizeResTextureFile();

    g_Sampler.Finalize( &g_Device );
    g_MatrixConstantBuffer.Finalize( &g_Device );
    g_ConstantBuffer.Finalize( &g_Device );
    g_IndexBuffer.Finalize( &g_Device );
    g_VertexBuffer.Finalize( &g_Device );

    g_RasterizerState.Finalize( &g_Device );
    g_BlendState.Finalize( &g_Device );
    g_DepthStencilState.Finalize( &g_Device );
    g_VertexState.Finalize( &g_Device );

    g_BufferDescriptorPool.Finalize( &g_Device );
    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    g_ViewportScissor.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    g_SwapChain.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPool[ idxMemoryPool ].Finalize( &g_Device );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    g_Device.Finalize();

    free( g_pMemoryHeap.Get() );
}


//-----------------------------------------------------------------------------
// HID

const nn::hid::NpadIdType g_NpadIds[] = {
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld,
    nn::hid::system::NpadSystemId::Other,
};
const int g_NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

// Npad のサンプリング間隔を返します
inline ::nn::TimeSpanType GetNpadSamplingInterval() NN_NOEXCEPT
{
    return ::nn::TimeSpanType::FromMilliSeconds(5);
}

// HID の初期化
void InitializeHid()
{
    nn::hid::InitializeNpad();

    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask);

    ::nn::hid::SetSupportedNpadIdType(g_NpadIds, g_NpadIdCountMax);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::NpadStateCountMax / 2 *
            GetNpadSamplingInterval().GetMilliSeconds()));
}

// HID の開放
void FinalizeHid()
{
}

// HID のフレーム処理
void TestHid()
{
    auto npadId = nn::hid::NpadId::Handheld;

    nn::hid::GetNpadState(&g_NpadHandheldState, npadId);
}


//-----------------------------------------------------------------------------
// オーディオ

struct AudioContext
{
    nn::mem::StandardAllocator allocator;
    nn::mem::StandardAllocator waveBufferAllocator;

    void* workBuffer;
    void* configBuffer;
    void* dataSine;

    nn::os::SystemEvent event;

    nn::audio::AudioRendererHandle handle;
    nn::audio::AudioRendererConfig config;

    nn::audio::VoiceType voiceSine;
};
AudioContext g_pAudioContext;
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_AudioWorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

// テスト用サイン波の生成
std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
    int16_t* p = static_cast<int16_t*>(g_pAudioContext.waveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

// オーディオの初期化
void InitializeAudio()
{
    const int32_t RenderRate = 32000;
    const int32_t RenderCount = RenderRate / 200;

    g_pAudioContext.allocator.Initialize(g_AudioWorkBuffer, sizeof(g_AudioWorkBuffer));
    g_pAudioContext.waveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2 + 0; // FinalMix(2) + SubMix(0)
    parameter.voiceCount = 24;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    g_pAudioContext.workBuffer = g_pAudioContext.allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_pAudioContext.workBuffer);

    // レンダラを初期化します。
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&g_pAudioContext.handle, &g_pAudioContext.event, parameter, g_pAudioContext.workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    g_pAudioContext.configBuffer = g_pAudioContext.allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_pAudioContext.configBuffer);
    nn::audio::InitializeAudioRendererConfig(&g_pAudioContext.config, parameter, g_pAudioContext.configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&g_pAudioContext.config, &finalMix, 2);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // オーディオ出力デバイスへの入力を設定します。
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    nn::Result result = nn::audio::AddDeviceSink(&g_pAudioContext.config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(g_pAudioContext.handle, &g_pAudioContext.config);
    NN_ABORT_UNLESS(result.IsSuccess());

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(g_pAudioContext.handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // Voice を用意します。
    nn::audio::WaveBuffer waveBufferSine;

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    bool rt = AcquireMemoryPool(&g_pAudioContext.config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // Voice で再生する波形を準備します。
    {
        const int sineSampleRate = 32000;
        const int sineFrequency = 440;
        const int sineSampleCount = 32000;
        std::size_t size = GenerateSineWave(&g_pAudioContext.dataSine, sineSampleRate, sineFrequency, sineSampleCount);

        nn::audio::AcquireVoiceSlot(&g_pAudioContext.config, &g_pAudioContext.voiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&g_pAudioContext.config, &g_pAudioContext.voiceSine, &finalMix);

        waveBufferSine.buffer = g_pAudioContext.dataSine;
        waveBufferSine.size = size;
        waveBufferSine.startSampleOffset = 0;
        waveBufferSine.endSampleOffset = sineSampleCount;
        waveBufferSine.loop = 0;
        waveBufferSine.isEndOfStream = false;

        // 波形を Voice に追加します。
        nn::audio::AppendWaveBuffer(&g_pAudioContext.voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&g_pAudioContext.voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&g_pAudioContext.voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&g_pAudioContext.voiceSine, &waveBufferSine);
        nn::audio::SetVoicePlayState(&g_pAudioContext.voiceSine, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&g_pAudioContext.voiceSine, &finalMix, 0.707f / 2, 0, 0);
    }
}

// オーディオの開放
void FinalizeAudio()
{
    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(g_pAudioContext.handle);
    nn::audio::CloseAudioRenderer(g_pAudioContext.handle);

    // メモリを解放します。
    if (g_pAudioContext.dataSine)
    {
        g_pAudioContext.waveBufferAllocator.Free(g_pAudioContext.dataSine);
        g_pAudioContext.dataSine = nullptr;
    }
    if (g_pAudioContext.configBuffer)
    {
        g_pAudioContext.allocator.Free(g_pAudioContext.configBuffer);
        g_pAudioContext.configBuffer = nullptr;
    }
    if (g_pAudioContext.workBuffer)
    {
        g_pAudioContext.allocator.Free(g_pAudioContext.workBuffer);
        g_pAudioContext.workBuffer = nullptr;
    }
}

// オーディオのフレーム処理
void TestAudio()
{
    nn::Result result;

    g_pAudioContext.event.Wait();
    result = nn::audio::RequestUpdateAudioRenderer(g_pAudioContext.handle, &g_pAudioContext.config);
    NN_ABORT_UNLESS(result.IsSuccess());
}


//-----------------------------------------------------------------------------
// アロケータ

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* GfxAllocate(size_t size, size_t alignment, void*)
{
    return aligned_alloc(alignment, size);
}
void GfxFree(void* addr, void*)
{
    free(addr);
}
void* GfxReallocate(void* addr, size_t newSize, void*)
{
    return realloc(addr, newSize);
}
#endif


void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* ptr, size_t)
{
    free(ptr);
}

}


//-----------------------------------------------------------------------------
// テスト

TEST(Integrate, Basic)
{
    nn::Result result;

    InitializeHid();
    InitializeAudio();

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();

    g_BufferDescriptorPool.BeginUpdate();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBuffer.GetGpuAddress(&gpuAddress);
        g_BufferDescriptorPool.SetBufferView(g_BufferDescriptorBaseIndex, gpuAddress, sizeof(float) * 4);

        g_MatrixConstantBuffer.GetGpuAddress(&gpuAddress);
        g_BufferDescriptorPool.SetBufferView(g_BufferDescriptorBaseIndex + 1, gpuAddress, sizeof(float) * 16);
    }
    g_BufferDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView(g_TextureDescriptorBaseIndex,
            g_pResTextureFile->GetResTexture(0)->GetTextureView());
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler(g_SamplerDescriptorBaseIndex, &g_Sampler);
    }
    g_SamplerDescriptorPool.EndUpdate();

    nn::audio::SetVoicePlayState(&g_pAudioContext.voiceSine, nn::audio::VoiceType::PlayState_Play);

    // 毎フレームのレンダリング
    for (int frame = 0; frame < 60 * 10; ++frame)
    {
        TestAudio();
        TestHid();

        MakeCommand(frame);

        UpdateGfx();

        // 定数バッファを更新
        float* pConstantBuffer = g_ConstantBuffer.Map< float >();
        {
            pConstantBuffer[0] = (frame & 0xFF) * (1.0f / 256.0f);
            pConstantBuffer[1] = 0.4f;
            pConstantBuffer[2] = 0.4f;
            pConstantBuffer[3] = 1.0f;
        }

        g_ConstantBuffer.Unmap();

        pConstantBuffer = g_MatrixConstantBuffer.Map< float >();
        {
            nn::util::Matrix4x3f mtx;
            nn::util::MatrixIdentity(&mtx);
            nn::util::MatrixSetTranslate(&mtx, g_Position);

            pConstantBuffer[ 0] = mtx.GetAxisX().GetX();
            pConstantBuffer[ 1] = mtx.GetAxisY().GetX();
            pConstantBuffer[ 2] = mtx.GetAxisZ().GetX();
            pConstantBuffer[ 3] = mtx.GetAxisW().GetX();
            pConstantBuffer[ 4] = mtx.GetAxisX().GetY();
            pConstantBuffer[ 5] = mtx.GetAxisY().GetY();
            pConstantBuffer[ 6] = mtx.GetAxisZ().GetY();
            pConstantBuffer[ 7] = mtx.GetAxisW().GetY();
            pConstantBuffer[ 8] = mtx.GetAxisX().GetZ();
            pConstantBuffer[ 9] = mtx.GetAxisY().GetZ();
            pConstantBuffer[10] = mtx.GetAxisZ().GetZ();
            pConstantBuffer[11] = mtx.GetAxisW().GetZ();
            pConstantBuffer[12] = 0;
            pConstantBuffer[13] = 0;
            pConstantBuffer[14] = 0;
            pConstantBuffer[15] = 1;
        }

        g_MatrixConstantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);

        // 結果の表示
        g_Queue.Present(&g_SwapChain, 1);
        g_Queue.Sync();
    }

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer(g_pLayer);
    nn::vi::CloseDisplay(g_pDisplay);
    nn::vi::Finalize();

    FinalizeAudio();
    FinalizeHid();

    SUCCEED();
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t mountRomCacheBufferSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize( &mountRomCacheBufferSize );
    NN_ASSERT( result.IsSuccess() );
    void* mountRomCacheBuffer = Allocate(mountRomCacheBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        Deallocate(mountRomCacheBuffer, mountRomCacheBufferSize);
    };

    result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, mountRomCacheBufferSize );
    NN_ASSERT( result.IsSuccess() );

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        nv::SetGraphicsAllocator(GfxAllocate, GfxFree, GfxReallocate, nullptr);
        nv::SetGraphicsDevtoolsAllocator(GfxAllocate, GfxFree, GfxReallocate, nullptr);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    auto testResult = RUN_ALL_TESTS();

    nn::fs::Unmount("Contents");

    nnt::Exit(testResult);
}
