﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UniformBufferManager.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that manages memory
 *  for uniform buffers by double buffering two chunks
 *  of GPU memory. The active chunk is written to and
 *  submitted to the GPU. At the end of the frame,
 *  it is swapped with the second chunk which then becomes
 *  active and writable for the next frame. This is done to
 *  avoid have to wait too much time for a sync object to
 *  signal that the memory is done being consumed by the
 *  GPU.
 */

#pragma once

#include <nvn\nvn.h>
#include <nvn\nvn_FuncPtrInline.h>
#include <vector>
#include <atomic>
#include "ManagedUniformBuffer.h"
#include "TutorialUtil.h"
#include "MemoryPool.h"

class UniformBufferManager
{
    public:
        UniformBufferManager(NVNdevice* pDevice, NVNqueue* pQueue, size_t chunkSize, int numChunks = 2);
        ~UniformBufferManager();
        ManagedUniformBuffer* CreateUniformBuffer(unsigned bufferSize);

        void SwapPools();
        void InsertFence();

        ptrdiff_t GetMemoryChunk(size_t size);

    private:
        int                                 m_NumChunks;
        MemoryPool                          m_Pool;
        NVNbuffer                           m_Buffer;
        size_t                              m_ChunkSize;
        size_t                              m_PoolSize;

        std::atomic<ptrdiff_t>              m_WriteIndex;
        unsigned                            m_CurrentChunk;
        NVNbufferAddress                    m_BufferAddress;
        char*                               m_pMappedBuffer;

        std::vector<ManagedUniformBuffer*>  m_ManagedBuffers;
        NVNsync*                            m_pSyncs;
        NVNdevice*                          m_pDevice;
        NVNqueue*                           m_pQueue;
        size_t                              m_bufferAlignment;
};
