﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <string>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace ncmutil {

class SubmissionPackageFileInstaller
{
public:
    nn::Result SetupBisWorkingDirectory(const char* directoryPath) NN_NOEXCEPT
    {
        auto workingPath = std::string(directoryPath) + "/work";
        nn::fs::DeleteDirectoryRecursively(workingPath.c_str());

        auto bisUserPath = workingPath + "/user";
        auto bisSystemPath = workingPath + "/system";

        NN_LOG("workingPath %s\n", workingPath.c_str());
        NN_LOG("bisUserPath %s\n", bisUserPath.c_str());
        NN_LOG("bisSystemPath %s\n", bisSystemPath.c_str());

        NN_RESULT_DO(nn::fs::CreateDirectory(workingPath.c_str()));
        NN_RESULT_DO(nn::fs::CreateDirectory(bisUserPath.c_str()));
        NN_RESULT_DO(nn::fs::CreateDirectory(bisSystemPath.c_str()));

        nn::fs::SetBisRootForHost(nn::fs::BisPartitionId::User, bisUserPath.c_str());
        nn::fs::SetBisRootForHost(nn::fs::BisPartitionId::System, bisSystemPath.c_str());

        m_WorkingRoot = workingPath;

        NN_RESULT_SUCCESS;
    }

    nn::Result TearDownBisWorkingDirectory() NN_NOEXCEPT
    {
        return nn::fs::DeleteDirectoryRecursively(m_WorkingRoot.c_str());
    }

    nn::Result InstallAll(std::vector<std::string>* outInstalled, const char* directoryPath, nn::ncm::StorageId storage) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, directoryPath, nn::fs::OpenDirectoryMode_File));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(dir); };

        std::vector<std::string> installedPathList;
        while (NN_STATIC_CONDITION(true))
        {
            int64_t readCount;
            nn::fs::DirectoryEntry entry;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readCount, &entry, dir, 1));
            if (readCount == 0)
            {
                break;
            }

            auto name = std::string(entry.name);
            auto ext = std::string(".nsp");
            if (name.substr(name.size() - ext.size()) == ext)
            {
                auto nspPath = std::string(directoryPath) + "/" + name;
                NN_RESULT_DO(InstallOne(nspPath.c_str(), storage));
                installedPathList.push_back(name);
            }
        }

        *outInstalled = installedPathList;

        NN_RESULT_SUCCESS;
    }

    nn::Result InstallAll(std::vector<std::string>* outInstalled, const char* directoryPath) NN_NOEXCEPT
    {
        return InstallAll(outInstalled, directoryPath, nn::ncm::StorageId::BuildInUser);
    }

    nn::Result InstallOne(const char* filePath, nn::ncm::StorageId storage) NN_NOEXCEPT
    {
        NN_LOG("Install %s.\n", filePath);

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

        static char g_Buffer[16 * 1024 * 1024];
        nn::ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, storage, g_Buffer, sizeof(g_Buffer)));
        NN_RESULT_DO(task.Prepare());
        auto progress = task.GetProgress();
        NN_LOG("Prepared. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Execute());
        progress = task.GetProgress();
        NN_LOG("Installed. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Commit());

        NN_RESULT_SUCCESS;
    }

    nn::Result InstallOne(const char* filePath) NN_NOEXCEPT
    {
        return InstallOne(filePath, nn::ncm::StorageId::BuildInUser);
    }

private:
    std::string m_WorkingRoot;
};

}}
