﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>

namespace nnt { namespace ldn
{
    /**
     * @brief       NIFM ライブラリを初期化します。
     */
    void InitializeNifm() NN_NOEXCEPT;

    /**
     * @brief       NIFM ライブラリの使用を終了します。
     */
    void FinalizeNifm() NN_NOEXCEPT;

    /**
     * @brief       無線スイッチの状態を変更します。
     * @param[in]   isEnabled       無線スイッチの有効・無効です。
     */
    void SetWirelessCommunicationSwitch(bool isEnabled) NN_NOEXCEPT;

    /**
     * @brief       一次的にスリープ状態に遷移させます。
     */
    void PutSleep() NN_NOEXCEPT;

    /**
     * @brief       スリープ状態から起床させます。
     */
    void WakeUp() NN_NOEXCEPT;

    /**
     * @brief       NIFM をロック状態に遷移させます。
     */
    void LockNifm() NN_NOEXCEPT;

    /**
     * @brief       NIFM のロック状態を解除します。
     */
    void UnlockNifm() NN_NOEXCEPT;

    /**
     * @brief       NIFM ライブラリの初期化を RAII で実施するヘルパです。
     */
    class NifmInitializer
    {
    public:

        /**
         * @brief       NIFM ライブラリを初期化します。
         */
        NifmInitializer() NN_NOEXCEPT
        {
            InitializeNifm();
        }

        /**
         * @brief       NIFM ライブラリを解放します。
         */
        ~NifmInitializer() NN_NOEXCEPT
        {
            FinalizeNifm();
        }
    };

    /**
     * @brief       一時的に無線スイッチの状態を変更するヘルパです。
     */
    class ScopedWirelessCommunicationSwitchSetter
    {
    public:

        /**
         * @brief       無線スイッチの状態を変更します。
         * @param[in]   isEnabled   無線スイッチの有効・無効です。
         */
        explicit ScopedWirelessCommunicationSwitchSetter(bool isEnabled) NN_NOEXCEPT
            : m_IsEnabled(isEnabled)
        {
            SetWirelessCommunicationSwitch(isEnabled);
        }

        /**
         * @brief       無線スイッチの状態を元に戻します。
         */
        ~ScopedWirelessCommunicationSwitchSetter() NN_NOEXCEPT
        {
            SetWirelessCommunicationSwitch(!m_IsEnabled);
        }

    private:

        bool m_IsEnabled;
    };

    /**
     * @brief       一時的に NIFM にスリープを模擬させるヘルパです。
     */
    class ScopedSleepSetter
    {
    public:

        /**
         * @brief       NIFM をスリープ状態に遷移させます。
         */
        explicit ScopedSleepSetter() NN_NOEXCEPT
        {
            PutSleep();
        }

        /**
         * @brief       スリープ状態を解除します。
         */
        ~ScopedSleepSetter() NN_NOEXCEPT
        {
            WakeUp();
        }
    };

    /**
     * @brief       一時的にロック状態に遷移させるヘルパです。
     */
    class ScopedNifmLock
    {
    public:

        /**
         * @brief       NIFM をロック状態に遷移させます。
         */
        explicit ScopedNifmLock() NN_NOEXCEPT
        {
            LockNifm();
        }

        /**
         * @brief       ロック状態を解除します。
         */
        ~ScopedNifmLock() NN_NOEXCEPT
        {
            UnlockNifm();
        }
    };

}} // namespace nnt::ldn
