﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/nn_Macro.h>

#include <nnt/graphics/testGraphics_AllocatorFunction.h>

namespace nnt{ namespace graphics{

    struct PngIhdr
    {
        uint32_t width;            //!< イメージの横幅
        uint32_t height;           //!<  イメージの縦幅
        uint8_t bitDepth;          //!<  ビット深度(1, 2, 4, 8, or 16 bits/channel)
        uint8_t colorType;         //!<  カラータイプ
        uint8_t compressionType;   //!<  圧縮手法
        uint8_t filterType;        //!<  フィルター手法
        uint8_t interlaceType;     //!<  インターレース手法
        uint8_t channels;          //!<  チャンネル数

        void SetDefault() NN_NOEXCEPT;

        /**
         * @brief ピクセル深度の取得
         *
         * @return     ピクセル深度を返します。
        */
        uint8_t GetPixelDepth() const NN_NOEXCEPT;

        /**
         * @brief 1行あたりのバイト数を取得
         *
         * @return     1行あたりのバイト数を返します。
        */
        uint32_t GetRowBytes() const NN_NOEXCEPT;
    };

    /**
        * @brief カラーマスク
        *
        * @details カラータイプの組み合わせ用
        */
    enum PngColorMask
    {
        PngColorMask_Palette = 1, //!< パレット使用
        PngColorMask_Color   = 2, //!< カラー
        PngColorMask_Alpha   = 4  //!< αチャンネル
    };

    /**
        * @brief カラータイプ
        *
        * @details 0 = グレイスケール
                3 = パレット
                2 = RGB
                6 = RGB+アルファ
                4 = グレイスケール+アルファ
        */
    enum PngColorType
    {
        PngColorType_Gray        = 0,                                            //!< グレイスケール
        PngColorType_Palette     = (PngColorMask_Color | PngColorMask_Palette),  //!< パレット
        PngColorType_Rgb         = (PngColorMask_Color),                         //!< RGB
        PngColorType_RgbAlpha    = (PngColorMask_Color | PngColorMask_Alpha),    //!< RGB+アルファ
        PngColorType_GrayAlpha   = PngColorMask_Alpha,                           //!< グレイスケール+アルファ
    };

    /**
    * @brief PNG入出力
    *
    * @details
    */
    class PngIO
    {
    public:


        /**
         * @brief PNG ファイル読み込み
         *
         * @param[out] pOutImageData     イメージデータ
         * @param[out] pOutImageDataSize イメージデータの大きさ（バイト）
         * @param[out] pOutIhdr          PNGのIHDRデータ
         * @param[in]  pPath             ファイルパス
         * @param[in]  allocateFunction  メモリの確保・解放を行う関数
         * @return     入力に成功したときtrueを返します。
        */
        static bool Read( uint8_t** pOutImageData, size_t* pOutImageDataSize, PngIhdr* pOutIhdr, const char* pPath, AllocatorFunction* allocateFunction );

        /**
         * @brief PNG ファイル書き込み
         *
         * @param[in] pPath            ファイルパス
         * @param[in] pImageData       イメージデータ
         * @param[in] imageDataSize    イメージデータの大きさ（バイト）
         * @param[in] pIhdr            PNGのIHDRデータ
         * @param[in] allocateFunction メモリの確保・解放を行う関数
         * @return    保存に成功したときtrueを返します。
        */
        static bool Write( const char* pPath, const uint8_t* pImageData, size_t imageDataSize, const PngIhdr* pIhdr, AllocatorFunction* allocateFunction);

        /**
         * @brief PNG の画像データを破棄します。
         *
         * @param[in,out] pImageData     破棄する画像のイメージデータ
         * @param[in,out] pImageDataSize 破棄する画像のイメージデータの大きさ（バイト）
         * @param[in] pIhdr      破棄する画像の IHDR データ
         * @param[in] allocateFunction 画像を作成した際に使用したアロケート関数
         * @details ReadPng() で確保したメモリを破棄します。
         */
        static void Destroy(uint8_t** pImageData, size_t* pImageDataSize, PngIhdr* pIhdr, AllocatorFunction* allocateFunction);
    };


}}
