﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/fsUtil/testFs_util_function.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief Write したデータを Read し、データの比較をします。
* @param[in]  file        ファイル操作のためのインターフェース
* @param[in]  writeBuffer 書込みデータ
* @param[in]  size        サイズ
* @retval     true        確認 ok の場合に返ります。
* @retval     false       確認 ng の場合に返ります。
* @details                Write、Flush、Read、比較の一連の処理を行うAPI
*/
template<typename FileClass>
void ConfirmWriteRead(FileClass* pFile, uint8_t *writeBuffer, const size_t size) NN_NOEXCEPT
{
    auto readBuffer = AllocateBuffer(size);

    FillBufferWithRandomValue(writeBuffer, size);

    NNT_EXPECT_RESULT_SUCCESS(pFile->Write(0, writeBuffer, size, nn::fs::WriteOption()));

    NNT_EXPECT_RESULT_SUCCESS(pFile->Flush());

    InvalidateVariable(readBuffer.get(), static_cast<int>(size));

    size_t resultSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(pFile->Read(&resultSize, 0, readBuffer.get(), size, nn::fs::ReadOption()));

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer.get(), size);
}

template<typename FileSystemClass>
bool IsEntryExist(FileSystemClass* pFilesSystem, const char* pPath, const nn::fs::DirectoryEntryType entryType) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType dirType;
    bool isGetEntryType;
    nn::Result entryResult;

    if( pFilesSystem != nullptr )
    {
        entryResult = pFilesSystem->GetEntryType(&dirType, pPath);
    }
    else
    {
        entryResult = nn::fs::GetEntryType(&dirType, pPath);
    }

    if( entryResult.IsSuccess() )
    {
        isGetEntryType = (dirType == entryType);
    }
    else
    {
        isGetEntryType = false;
    }

    return isGetEntryType;
}

/**
* @brief 指定エントリタイプが存在するか確認する関数です
* @param[in]  pFileSystem            ファイルシステム
* @param[in]  name                   確認するパスを指定します
* @param[in]  entryType              確認するファイル／ディレクトリのエントリタイプを指定します
*
* @return     指定エントリタイプの結果が返ります。
* @retval     true                   指定のエントリタイプで存在した場合に返ります。
* @retval     false                  指定のエントリタイプで存在しない場合に返ります。
*/
template<typename FileSystemClass>
bool IsEntryExist(FileSystemClass* pFileSystem, const String name, const nn::fs::DirectoryEntryType entryType) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entry;

    // 指定パスのエントリタイプを取得します
    if( pFileSystem->GetEntryType(&entry, name.c_str()).IsFailure() )
    {
        return false;
    }

    // 取得したエントリタイプの識別を判定します
    return ((entry & entryType) == entryType);
}

/**
* @brief テスト対象のファイルを作成します。
* @param[in]  fileSystem ファイルシステム
* @param[out] pFilePath  ファイルパスの配列（nullptr を指定する場合は、作成のみを行い、パスの配列は返さない。）
* @param[in]  pPath      対象のベースパス
* @param[in]  count      作成数
* @param[in]  size       サイズ
* @details               作成場所(パス)、作成数、サイズを指定し、ファイルの作成処理を行う API
*/
template<typename FileSystemClass>
nn::Result CreateTestFile(FileSystemClass* pFileSystem, char(*pFilePath)[NameMaxLength], const char* pPath, const int count, const int64_t size) NN_NOEXCEPT
{
    auto workPath = AllocateBuffer(NameMaxLength);
    for( int32_t i = 0; i < count; i++ )
    {
        nn::Result ret = nn::fs::ResultUnknown();
        nn::util::SNPrintf(workPath.get(), NameMaxLength, "%s%04d.file", pPath, i);
        if( pFileSystem == nullptr )
        {
            ret = nn::fs::CreateFile(workPath.get(), size);
        }
        else
        {
            ret = pFileSystem->CreateFile(workPath.get(), size);
        }

        if( ret.IsFailure() )
        {
            NN_LOG("loop count: %d, file path: %s\n", i, workPath.get());
            NN_RESULT_THROW(ret);
        }

        if( pFilePath != nullptr )
        {
            std::strncpy(pFilePath[i], workPath.get(), NameMaxLength);
        }
    }
    NN_RESULT_SUCCESS;
}

inline nn::Result CreateTestFile(nullptr_t pFileSystem, char(*pFilePath)[NameMaxLength], const char* pPath, const int count, const int64_t size) NN_NOEXCEPT
{
    return CreateTestFile<nn::fs::fsa::IFileSystem>(pFileSystem, pFilePath, pPath, count, size);
}

/**
* @brief テスト対象のディレクトリを作成します。
* @param[in]  pFileSystem  ファイルシステム
* @param[out] pDirPath     ディレクトリパスの配列（nullptr を指定する場合は、作成のみを行い、パスの配列は返さない。）
* @param[in]  pPath        対象のベースパス
* @param[in]  count        作成数
* @param[in]  type         構成 (階層構造又は、フラット)
*                            DIRTYPE_FLAT : 同ディレクトリ内にcount数の子ディレクトリ
*                            DIRTYPE_NEST : count階層の子ディレクトリ
* @param[in]  isFileCreate ファイル作成の有無
* @param[in]  size         ファイルサイズ
* @details                 作成場所(パス)、作成数、構成(階層又は、フラット)、ファイル作成の有無・サイズを指定し、フォルダの作成処理を行う API
*/
template<typename FileSystemClass>
nn::Result CreateTestDirectory(FileSystemClass* pFileSystem, char(*pDirPath)[NameMaxLength],
    const char* pPath, const int count, const int type,
    const bool isFileCreate, const int size) NN_NOEXCEPT
{
    auto workPath = AllocateBuffer(NameMaxLength);
    nn::Result ret = nn::fs::ResultUnknown();
    String dirPath = pPath;
    String filePath;

    for( int i = 0; i < count; i++ )
    {
        switch( type )
        {
        case DIRTYPE_FLAT:
        {
            nn::util::SNPrintf(workPath.get(), NameMaxLength, "%s%04d/", pPath, i);
            if( pFileSystem == nullptr )
            {
                ret = nn::fs::CreateDirectory(workPath.get());
            }
            else
            {
                ret = pFileSystem->CreateDirectory(workPath.get());
            }
            filePath = workPath.get();

            if( ret.IsFailure() )
            {
                NN_LOG("loop count: %d, directory path: %s\n", i, workPath.get());
                return nn::fs::ResultUnknown();
            }
        }
        break;

        case DIRTYPE_NEST:
        {
            nn::util::SNPrintf(workPath.get(), NameMaxLength, "%d/", i % 10);
            dirPath += workPath.get();
            if( pFileSystem == nullptr )
            {
                ret = nn::fs::CreateDirectory(dirPath.c_str());
            }
            else
            {
                ret = pFileSystem->CreateDirectory(dirPath.c_str());
            }
            filePath = dirPath;

            if( ret.IsFailure() )
            {
                NN_LOG("loop count: %d, directory path: %s\n", i, dirPath.c_str());
                break;
            }
        }
        break;

        default:
        {
            NN_LOG("\n***Error Case***\n");
            return nn::fs::ResultUnknown();
        }
        break;
        }

        if( pDirPath != nullptr )
        {
            std::strncpy(pDirPath[i], filePath.c_str(), NameMaxLength);
        }

        if( isFileCreate == true )
        {
            nn::util::SNPrintf(workPath.get(), NameMaxLength, "test.file");
            filePath += workPath.get();
            if( pFileSystem == nullptr )
            {
                ret = CreateFileWith8BitCount(filePath.c_str(), size, 0);
            }
            else
            {
                ret = pFileSystem->CreateFile(filePath.c_str(), size);
            }

            if( ret.IsFailure() )
            {
                NN_LOG("loop count: %d, file path: %s\n", i, filePath.c_str());
                return nn::fs::ResultUnknown();
            }
        }
    }
    return ret;
}

inline nn::Result CreateTestDirectory(std::nullptr_t pFileSystem, char(*pDirPath)[NameMaxLength],
    const char* pPath, const int count, const int type,
    const bool isFileCreate, const int size) NN_NOEXCEPT
{
    return CreateTestDirectory<nn::fs::fsa::IFileSystem>(pFileSystem, pDirPath, pPath, count, type, isFileCreate, size);
}

}}}
