﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/nn_SdkLog.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief    書き込みにおけるサイズをチェックするストレージ
*/
class WriteSizeCheckStorage : public nn::fs::IStorage
{
public:
    /**
    * @brief        コンストラクタです。
    *
    * @param[in]    pBaseStorage        基礎となるストレージ
    * @param[in]    alignment           オフセットのアライメント
    * @param[in]    size                書き込みサイズ
    * @param[in]    makeCheckStorage    チェックするストレージを作成する関数
    *
    * @pre
    *               - pBaseStorage != nullptr
    *
    * @details      makeCheckStorage は基礎ストレージを IStorage* で受け取り、
    *               チェックするストレージを std::unique_ptr<IStorage> にして返します。
    *               このように作成されたストレージが下位ストレージに書き込みを行う際、
    *               先頭の alignment アラインされていない領域および末尾を除いて
    *               書き込みサイズが size に等しいことを確認します。
    */
    template<typename Functor>
    WriteSizeCheckStorage(
        nn::fs::IStorage* pBaseStorage,
        size_t alignment,
        size_t size,
        Functor makeCheckStorage
    ) NN_NOEXCEPT
        : m_CoreStorage(pBaseStorage, alignment, size),
          m_pCheckStorage()
    {
        m_pCheckStorage = makeCheckStorage(&m_CoreStorage);
    }

    virtual ~WriteSizeCheckStorage() NN_NOEXCEPT NN_OVERRIDE {}

    void SetChecked(bool isChecked) NN_NOEXCEPT
    {
        m_CoreStorage.SetChecked(isChecked);
    }

    virtual nn::Result Read(
                int64_t offset,
                void* buffer,
                size_t size
            ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCheckStorage->Read(offset, buffer, size);
    }

    /**
    * @brief    サイズをチェックしてから書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return   関数の処理結果を返します。
    * @retval   ResultInvalidSize   不正なサイズで書き込みが行われました。
    *
    * @details  書き込みの先頭のアライメントされていない範囲および末尾以外で
    *           コンストラクタで指定されたサイズでないサイズの書き込みを行っていることを確認します。
    */
    virtual nn::Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        m_CoreStorage.SetWriteRange(offset, size);
        return m_pCheckStorage->Write(offset, buffer, size);
    }

    virtual nn::Result OperateRange(
                void* outBuffer,
                size_t outBufferSize,
                nn::fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCheckStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

    using IStorage::OperateRange;

    virtual nn::Result GetSize(int64_t* outSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCheckStorage->GetSize(outSize);
    }

    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCheckStorage->SetSize(size);
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pCheckStorage->Flush();
    }

private:
    class WriteSizeCheckCoreStorage : public nn::fs::IStorage
    {
    public:
        WriteSizeCheckCoreStorage(
            nn::fs::IStorage* pBaseStorage,
            size_t alignment,
            size_t size
        ) NN_NOEXCEPT
            : m_pBaseStorage(pBaseStorage),
              m_CheckAlignment(alignment),
              m_CheckSize(size)
        {
            NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
        }

        void SetWriteRange(int64_t offset, size_t size) NN_NOEXCEPT
        {
            m_WriteOffset = offset;
            m_WriteSize = size;
        }

        void SetChecked(bool isChecked) NN_NOEXCEPT
        {
            m_IsChecked = isChecked;
        }

        virtual nn::Result Read(
                               int64_t offset,
                               void* buffer,
                               size_t size
                           ) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->Read(offset, buffer, size);
        }

        virtual nn::Result Write(
                               int64_t offset,
                               const void* buffer,
                               size_t size
                           ) NN_NOEXCEPT NN_OVERRIDE
        {
            // 書き込み範囲の先頭の非アライン領域または末尾でないなら指定のサイズで書き込まなければならない
            if( m_IsChecked
                && (offset != m_WriteOffset || nn::util::is_aligned(offset, m_CheckAlignment))
                && offset + size != m_WriteOffset + m_WriteSize
                && size != m_CheckSize )
            {
                return nn::fs::ResultInvalidSize();
            }
            return m_pBaseStorage->Write(offset, buffer, size);
        }

        virtual nn::Result OperateRange(
                               void* outBuffer,
                               size_t outBufferSize,
                               nn::fs::OperationId operationId,
                               int64_t offset,
                               int64_t size,
                               const void* inBuffer,
                               size_t inBufferSize
                           ) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize);
        }

        virtual nn::Result GetSize(int64_t* outSize) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->GetSize(outSize);
        }

        virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->SetSize(size);
        }

        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->Flush();
        }

    private:
        nn::fs::IStorage* m_pBaseStorage;
        size_t m_CheckAlignment;
        size_t m_CheckSize;
        int64_t m_WriteOffset;
        size_t m_WriteSize;
        bool m_IsChecked;
    };

private:
    WriteSizeCheckCoreStorage m_CoreStorage;
    std::unique_ptr<nn::fs::IStorage> m_pCheckStorage;
};

}}}
