﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief 既に初期化されたかどうかを調べつつ読み書きするストレージクラスです。
*/
class WriteCheckStorage : public nn::fs::IStorage
{
private:
    Vector<uint8_t> m_WriteFlags;
    nn::fs::IStorage* m_pStorage;
    size_t m_Size;        //!< ストレージのサイズ

public:
    /**
    * @brief コンストラクタです。
    *
    */
    explicit WriteCheckStorage()
        : m_pStorage(nullptr)
        , m_Size(0)
    {
    }

    /**
    * @breif ストレージをマウントします。
    */
    virtual nn::Result Initialize(IStorage *pStorage)
    {
        NN_SDK_ASSERT_NOT_NULL(pStorage);

        int64_t size;
        nn::Result result = pStorage->GetSize(&size);
        NN_ASSERT(result.IsSuccess());
        m_pStorage = pStorage;

        m_Size = static_cast<size_t>(size);
        size_t sizeBits = (m_Size + 7) / 8;
        m_WriteFlags.resize(sizeBits);

        uint8_t* p = &m_WriteFlags[0];
        std::memset(p, 0, sizeBits);

        return  nn::ResultSuccess();
    }

    /**
    * @breif ストレージのサイズを返します。
    */
    virtual nn::Result GetSize(int64_t* pOut) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_SDK_ASSERT_NOT_NULL(pOut);
        *pOut = m_Size;
        return nn::ResultSuccess();
    }

    /**
    * @brief 下位ストレージの offset から size バイト読み込み buffer にコピーします。
    */
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT(m_Size > 0);
        NN_SDK_ASSERT(offset + size <= m_Size);

        // 既に書き込みされた領域か判定します。
        {
            const uint8_t* pWritten = &m_WriteFlags[0];
            size_t sizeCheck = size;
            size_t offsetCheck = static_cast<size_t>(offset);
            while( sizeCheck != 0 )
            {
                // 1bitずつ確認していきます。
                if( !(pWritten[offsetCheck / 8] & (1 << (offsetCheck & 7))) )
                {
                    // TODO 差し替え
//                    return nn::fs0::ResultAccessDenied();
                    return nn::fs::ResultUnexpected();
                }
                offsetCheck ++;
                sizeCheck --;
            }
        }

        return m_pStorage->Read(offset, buffer, size);
    }

    /**
    * @brief 下位ストレージの offset 以降に buffer を size バイト分コピーします。
    */
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT(m_Size > 0);
        NN_SDK_ASSERT(offset + size <= m_Size);

        nn::Result result = m_pStorage->Write(offset, buffer, size);

        // 書き込みが成功したので、この範囲の読み込みは正しく行えるものとします。
        if( !result.IsFailure() )
        {
            uint8_t* pWritten = &m_WriteFlags[0];
            size_t offsetWritten = static_cast<size_t>(offset);
            while( size )
            {
                // 1bitずつマークしていきます。
                pWritten[offsetWritten / 8] |= (1 << (offsetWritten & 7));
                offsetWritten ++;
                size --;
            }
        }
        return result;
    }

    /**
     * @brief   フラッシュします。
     *
     * @return  関数の処理結果を返します。
     * @retval  ResultSuccess   必ず成功します。
     */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
};

}}}
