﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <algorithm>
#include <atomic>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief データをメモリで保持するファイルクラスです。
*/
class MemoryFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(MemoryFile);

public:
    /**
    * @brief  コンストラクタです。
    */
    MemoryFile() NN_NOEXCEPT
        : m_Memory()
    {
    }

    /**
    * @brief    デストラクタです。
    */
    virtual ~MemoryFile() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[out]   outValue  読み込んだサイズ
    * @param[in]    offset    読み込み開始位置
    * @param[out]   buffer    読み込んだ内容をコピーするバッファ
    * @param[in]    size      読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result DoRead(
        size_t* outValue,
        int64_t offset,
        void* buffer,
        size_t size,
        const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_UNUSED(option);
        if( static_cast<int64_t>(m_Memory.size()) < offset )
        {
            NN_RESULT_THROW(nn::fs::ResultOutOfRange());
        }
        *outValue = offset + size < m_Memory.size()
            ? size : static_cast<size_t>(m_Memory.size() - offset);
        std::copy(
            std::next(m_Memory.begin(), static_cast<size_t>(offset)),
            std::next(m_Memory.begin(), static_cast<size_t>(offset + *outValue)),
            reinterpret_cast<char*>(buffer));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    * @param[in]    option  書き込みオプション
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result DoWrite(
        int64_t offset,
        const void* buffer,
        size_t size,
        const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(option);
        if( static_cast<int64_t>(m_Memory.size()) < offset )
        {
            NN_RESULT_THROW(nn::fs::ResultOutOfRange());
        }
        if( m_Memory.size() < offset + size )
        {
            m_Memory.resize(static_cast<size_t>(offset + size));
        }
        std::copy(
            reinterpret_cast<const char*>(buffer),
            reinterpret_cast<const char*>(buffer) + size,
            std::next(m_Memory.begin(), static_cast<size_t>(offset)));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief    フラッシュします。
    *
    * @return   関数の処理結果を返します。
    */
    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルサイズを設定します。
    *
    * @param[in]    size    新しいサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        m_Memory.resize(static_cast<size_t>(size));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    取得したサイズ格納先
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        *outValue = static_cast<int64_t>(m_Memory.size());
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        特定のメモリ領域に既定の操作を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     実行する操作
    * @param[in]    offset          メモリ領域の開始位置
    * @param[in]    size            メモリ領域のサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return  関数の処理結果を返します。
    */
    virtual nn::Result DoOperateRange(
        void*, size_t, nn::fs::OperationId, int64_t, int64_t, const void*, size_t
    ) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

private:
    Vector<char> m_Memory;
};

}}} // namespace nnt::fs::util
