﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

#include <nn/nn_Log.h>
#include <nnt/codecUtil/testCodec_TestBase.h>
#include <nnt/codecUtil/testCodec_ObjectBase.h>

namespace nnt {
namespace codec {

// Derive and use
class DecodedDataBufferBase
    : protected ObjectBase
{

protected:
    DecodedDataBufferBase() NN_NOEXCEPT
        : m_DecodedDataBufferAddress(nullptr)
        , decodedDataBuffer(nullptr)
        , decodedDataBufferSize(0)
        , decodedSampleCount(0)
    {}

    virtual ~DecodedDataBufferBase() NN_NOEXCEPT
    {
        if (nullptr != m_DecodedDataBufferAddress)
        {
            std::free(m_DecodedDataBufferAddress);
        }
        m_DecodedDataBufferAddress = nullptr;
        decodedDataBuffer = nullptr;
    }

    void Initialize(const TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        // Define some aliases
        const int& sampleRate = parameterSet.sampleRate;
        const int& channelCount = parameterSet.channelCount;
        const int& frame = parameterSet.frame;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        NN_ASSERT(frame != 0);
        // Initializing
        decodedDataBufferSize = static_cast<std::size_t>(frame) * sampleRate / 1000 / 1000 * channelCount * sizeof(int16_t);
        m_DecodedDataBufferAddress = std::malloc(decodedDataBufferSize);
        ASSERT_NE(nullptr, m_DecodedDataBufferAddress);
        decodedDataBuffer = static_cast<int16_t*>(m_DecodedDataBufferAddress);
        // Done
        m_IsInitialized = true;
    }

    void Finalize() NN_NOEXCEPT
    {
        m_IsInitialized = false;
        if (nullptr != m_DecodedDataBufferAddress)
        {
            std::free(m_DecodedDataBufferAddress);
        }
        m_DecodedDataBufferAddress = nullptr;
        decodedDataBuffer = nullptr;
        decodedDataBufferSize = 0;
        decodedSampleCount = 0;
    }

private:
    void* m_DecodedDataBufferAddress;

protected:
    int16_t* decodedDataBuffer;
    std::size_t decodedDataBufferSize;
    int decodedSampleCount;
};

}} // nnt::codec
