﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_BitTypes.h>
#include <nnt/nntest.h>

#include <nn/image/image_ExifBuilder.h>
#include <nn/image/image_ExifExtractor.h>
#include <image_ExifParser.h>

#include "testImageJpeg_Io.h"

/**
    @file nn::image::ExifBuilder のテスト
    @details
    ExifExtractor はテスト済みとする。
 */

TEST(ImageExifBuilder, AssertionTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::ExifBuilder pre-condition check\n");

    size_t workBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
    void *workBuf = malloc(workBufSize);

    NN_LOG("  - Test ID: %s\n", "Constructor");
    EXPECT_DEATH_IF_SUPPORTED({nn::image::ExifBuilder builder(nullptr, 0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({nn::image::ExifBuilder builder(workBuf, workBufSize - 1);}, ".*");

    nn::image::ExifBuilder builder(workBuf, workBufSize);

    NN_LOG("  - Test ID: %s\n", "Set****");
    const unsigned char data[] = {0xFF, 0x00};
    EXPECT_DEATH_IF_SUPPORTED({builder.SetOrientation((nn::image::ExifOrientation)0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetOrientation((nn::image::ExifOrientation)9);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetSoftware("a", 0);}, ".*"); // サイズがそもそも短い
    EXPECT_DEATH_IF_SUPPORTED({builder.SetSoftware("a", 3);}, ".*"); // 途中で \0
    EXPECT_DEATH_IF_SUPPORTED({builder.SetSoftware("a !", 3);}, ".*"); // 終端されていない
    EXPECT_DEATH_IF_SUPPORTED({builder.SetSoftware((const char*)data, sizeof(data));}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetDateTime("    :  :     :  : ", 19);}, ".*"); // サイズがそもそも短い。
    EXPECT_DEATH_IF_SUPPORTED({builder.SetDateTime("    :  :     :  : ", 20);}, ".*"); // 途中で \0
    EXPECT_DEATH_IF_SUPPORTED({builder.SetDateTime("    :  :     :  :  !", 20);}, ".*"); // 終端されていない
    EXPECT_DEATH_IF_SUPPORTED({builder.SetDateTime((const char*)data, 20);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetMakerNote(data, 0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetUniqueId("000102030405060708090A0B0C0D0E0", 32);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetUniqueId("000102030405060708090A0B0C0D0E0", 33);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetUniqueId("000102030405060708090A0B0C0D0E0F!", 33);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetUniqueId((const char*)data, 33);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.SetThumbnail(data, 0);}, ".*");

    NN_LOG("  - Test ID: %s\n", "Un-analyzed death");
    EXPECT_DEATH_IF_SUPPORTED({builder.GetAnalyzedOutputSize();}, ".*");
    void *outBuf = malloc(0xFFF7);
    nn::image::Dimension dim = {0x01, 0x02};
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, dim);}, ".*");

    NN_LOG("  - Test ID: %s\n", "Build");
    ASSERT_EQ(nn::image::JpegStatus_Ok, builder.Analyze());
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(nullptr, 0xFFF7, dim);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0, dim);}, ".*");
    nn::image::Dimension ngDimX = {0x1FFFF, 0x01};
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, ngDimX);}, ".*");
    nn::image::Dimension ngDimX2 = {0x00, 0x01};
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, ngDimX2);}, ".*");
    nn::image::Dimension ngDimY = {0x01, 0x1FFFF};
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, ngDimY);}, ".*");
    nn::image::Dimension ngDimY2 = {0x01, 0x00};
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, ngDimY2);}, ".*");

    NN_LOG("  - Test ID: %s\n", "Un-analyzed death (again with state rollback)");
    builder.SetOrientation(nn::image::ExifOrientation_Normal);
    EXPECT_DEATH_IF_SUPPORTED({builder.GetAnalyzedOutputSize();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({builder.Build(outBuf, 0xFFF7, dim);}, ".*");

    free(outBuf);
    free(workBuf);
} // NOLINT(readability/fn_size)

TEST(ImageExifBuilder, BuildTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::ExifBuilder\n");

    // U-B-01
    {
        NN_LOG(" - Test ID: %s\n", "U-B-01");
        // 実効画像サイズのみを指定した場合の出力の正解データ
        static const uint8_t truth[] = {
            0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08, 0x00, 0x06, 0x01, 0x0F, 0x00, 0x02, 0x00, 0x00,
            0x00, 0x12, 0x00, 0x00, 0x00, 0x56, 0x01, 0x1A, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
            0x00, 0x68, 0x01, 0x1B, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x70, 0x01, 0x28,
            0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x02, 0x13, 0x00, 0x03, 0x00, 0x00,
            0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x87, 0x69, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
            0x00, 0x78, 0x00, 0x00, 0x00, 0x00, 0x4E, 0x69, 0x6E, 0x74, 0x65, 0x6E, 0x64, 0x6F, 0x20, 0x63,
            0x6F, 0x2E, 0x2C, 0x20, 0x6C, 0x74, 0x64, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01,
            0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x06, 0x90, 0x00, 0x00, 0x07, 0x00, 0x00,
            0x00, 0x04, 0x30, 0x32, 0x33, 0x30, 0x91, 0x01, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x01, 0x02,
            0x03, 0x00, 0xA0, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x30, 0x31, 0x30, 0x30, 0xA0, 0x01,
            0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0xA0, 0x02, 0x00, 0x03, 0x00, 0x00,
            0x00, 0x01, 0x01, 0x23, 0x00, 0x00, 0xA0, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0xCD, 0xEF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00
        };

        size_t workBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
        void *workBuf = malloc(workBufSize);
        nn::image::ExifBuilder builder(workBuf, workBufSize);

        ASSERT_EQ(nn::image::JpegStatus_Ok, builder.Analyze());

        size_t outSize = builder.GetAnalyzedOutputSize();
        ASSERT_EQ(sizeof(truth), outSize);

        nn::image::Dimension dim = {0x0123, 0xCDEF};
        void *outBuf = malloc(outSize);
        builder.Build(outBuf, outSize, dim);
        free(workBuf);

        EXPECT_EQ(0, std::memcmp(truth, outBuf, outSize));
        free(outBuf);
    }

    // U-B-02
    {
        NN_LOG(" - Test ID: %s\n", "U-B-02");

        // Extractor 初期化
        size_t workBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
        void *workBuf = malloc(workBufSize);
        nn::image::ExifBuilder builder(workBuf, workBufSize);

        // メタ情報の設定
        static const char software[] = "A";
        static const char dateTime[] = "    :  :     :  :  ";
        NN_STATIC_ASSERT(sizeof(dateTime) == 20);
        static const uint8_t makerNote[] = {0xFA};;
        static const char uniqueId[] = "000102030405060708090A0B0C0D0E0F";
        NN_STATIC_ASSERT(sizeof(uniqueId) == 33);
        static const uint8_t thumbnail[] = {0xFF, 0xD8, 0xFF, 0xD9};
        builder.SetOrientation(nn::image::ExifOrientation_Normal);
        builder.SetSoftware(software, sizeof(software));
        builder.SetDateTime(dateTime, sizeof(dateTime));
        builder.SetMakerNote(makerNote, sizeof(makerNote));
        builder.SetUniqueId(uniqueId, sizeof(uniqueId));
        builder.SetUniqueId(nullptr, 0); // ユニーク ID 取り消し
        builder.SetThumbnail(thumbnail, sizeof(thumbnail));
        builder.SetThumbnail(nullptr, 0); // サムネイル取り消し

        // 生成
        ASSERT_EQ(nn::image::JpegStatus_Ok, builder.Analyze());
        size_t outSize = builder.GetAnalyzedOutputSize();
        ASSERT_GE(0xFFF7, static_cast<int>(outSize));

        nn::image::Dimension dim = {0x0123, 0xCDEF};
        void *outBuf = malloc(outSize);
        builder.Build(outBuf, outSize, dim);
        free(workBuf);

        // 値のチェック
        size_t extBufSize = nn::image::ExifExtractor::GetWorkBufferSize();
        void *extBuf = malloc(extBufSize);
        nn::image::ExifExtractor extractor(extBuf, extBufSize);
        extractor.SetExifData(outBuf, outSize);
        ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

        const char *strPtr;
        size_t size;
        //// 0th TIFF
        strPtr = extractor.ExtractMaker(&size);
        EXPECT_EQ(0, std::memcmp("Nintendo co., ltd", strPtr, sizeof("Nintendo co., ltd")));
        EXPECT_EQ(nullptr, extractor.ExtractModel(&size));
        nn::image::ExifOrientation orientation;
        EXPECT_TRUE(extractor.ExtractOrientation(&orientation));
        EXPECT_EQ(nn::image::ExifOrientation_Normal, orientation);
        strPtr = extractor.ExtractSoftware(&size);
        EXPECT_EQ(0, std::memcmp(software, strPtr, sizeof(software)));
        strPtr = extractor.ExtractDateTime(&size);
        EXPECT_EQ(0, std::memcmp(dateTime, strPtr, sizeof(dateTime)));

        //// 0th Exif
        const void *dataPtr;
        dataPtr = extractor.ExtractMakerNote(&size);
        EXPECT_EQ(0, std::memcmp(makerNote, dataPtr, sizeof(makerNote)));
        nn::image::Dimension extDim;
        EXPECT_TRUE(extractor.ExtractEffectiveDimension(&extDim));
        EXPECT_EQ(0x0123, extDim.width);
        EXPECT_EQ(0xCDEF, extDim.height);
        EXPECT_EQ(nullptr, extractor.ExtractUniqueId(&size));

        //// 1st TIFF
        EXPECT_EQ(nullptr, extractor.ExtractThumbnail(&size));

        free(outBuf);
        free(extBuf);
    }

    // U-B-03
    {
        NN_LOG(" - Test ID: %s\n", "U-B-03");

        // Extractor 初期化
        size_t workBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
        void *workBuf = malloc(workBufSize);
        nn::image::ExifBuilder builder(workBuf, workBufSize);

        // メタ情報の設定
        static const char software[] = " !\"#$%&'()*+`-./:;<=>?@[\\]^_`{|}~";
        static const char dateTime[] = "2015:01:01 00:00:59";
        NN_STATIC_ASSERT(sizeof(dateTime) == 20);
        static const uint8_t makerNote[0x8000] = {};;
        static const char uniqueId[] = "000102030405060708090A0B0C0D0E0F";
        NN_STATIC_ASSERT(sizeof(uniqueId) == 33);
        static const uint8_t thumbnail[0x7E40] = {};
        builder.SetOrientation(nn::image::ExifOrientation_Rotate90);
        builder.SetSoftware(software, sizeof(software));
        builder.SetDateTime(dateTime, sizeof(dateTime));
        builder.SetMakerNote(makerNote, sizeof(makerNote));
        builder.SetUniqueId(uniqueId, sizeof(uniqueId));
        builder.SetThumbnail(thumbnail, sizeof(thumbnail));

        // 生成
        ASSERT_EQ(nn::image::JpegStatus_Ok, builder.Analyze());
        size_t outSize = builder.GetAnalyzedOutputSize();
        ASSERT_EQ(0xFFF7, static_cast<int>(outSize));

        nn::image::Dimension dim = {0xEF12, 0x34CD};
        void *outBuf = malloc(outSize);
        builder.Build(outBuf, outSize, dim);
        free(workBuf);

        // 値のチェック
        size_t extBufSize = nn::image::ExifExtractor::GetWorkBufferSize();
        void *extBuf = malloc(extBufSize);
        nn::image::ExifExtractor extractor(extBuf, extBufSize);
        extractor.SetExifData(outBuf, outSize);
        ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

        const char *strPtr;
        size_t size;
        //// 0th TIFF
        strPtr = extractor.ExtractMaker(&size);
        EXPECT_EQ(0, std::memcmp("Nintendo co., ltd", strPtr, sizeof("Nintendo co., ltd")));
        EXPECT_EQ(nullptr, extractor.ExtractModel(&size));
        nn::image::ExifOrientation orientation;
        EXPECT_TRUE(extractor.ExtractOrientation(&orientation));
        EXPECT_EQ(nn::image::ExifOrientation_Rotate90, orientation);
        strPtr = extractor.ExtractSoftware(&size);
        EXPECT_EQ(0, std::memcmp(software, strPtr, sizeof(software)));
        strPtr = extractor.ExtractDateTime(&size);
        EXPECT_EQ(0, std::memcmp(dateTime, strPtr, sizeof(dateTime)));

        //// 0th Exif
        const void *dataPtr;
        dataPtr = extractor.ExtractMakerNote(&size);
        EXPECT_EQ(0, std::memcmp(makerNote, dataPtr, sizeof(makerNote)));
        nn::image::Dimension extDim;
        EXPECT_TRUE(extractor.ExtractEffectiveDimension(&extDim));
        EXPECT_EQ(0xEF12, extDim.width);
        EXPECT_EQ(0x34CD, extDim.height);
        strPtr = extractor.ExtractUniqueId(&size);
        EXPECT_EQ(0, std::memcmp(uniqueId, strPtr, size));

        //// 1st TIFF
        dataPtr = extractor.ExtractThumbnail(&size);
        EXPECT_EQ(0, std::memcmp(thumbnail, dataPtr, size));

        free(outBuf);
        free(extBuf);
    }

    // U-B-99
    {
        NN_LOG(" - Test ID: %s\n", "U-B-99");

        // Extractor 初期化
        size_t workBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
        void *workBuf = malloc(workBufSize);
        nn::image::ExifBuilder builder(workBuf, workBufSize);

        // メタ情報の設定
        static const uint8_t makerNote[0xFF26] = {};;
        builder.SetMakerNote(makerNote, sizeof(makerNote));

        // 生成
        ASSERT_EQ(nn::image::JpegStatus_OutputOverabounds, builder.Analyze());

        free(workBuf);
    }
} // NOLINT(readability/fn_size)
