﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <csetjmp>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nnt/nntest.h>

// import from External/libjpeg/include
#include <jpeglib.h>
#include <jerror.h>

#include "testImageJpeg_LibjpegApi.h"

namespace libjpeg = nn::image::detail::jpeg;

namespace nnt { namespace image { namespace jpeg {

namespace
{
/**
    @brief Libjpeg内部で発生したエラーを処理するための諸々を保持しておく構造体。
 */
struct ErrorContext
{
    struct nn::image::detail::jpeg::jpeg_error_mgr errMgr;  ///< libjpeg内部のエラー構造体。無理やりキャストして使うので、必ず先頭に置くこと。
    jmp_buf jmpBuf; ///< setjmp でエラー発生時の戻り先を保持しておく。
    volatile bool isExpected; ///< エラーが発生したときに、それが期待通りかを返す。
    volatile ErrorHandler handler; ///< エラー値が期待通りかを評価する関数。
};

static void ErrorExit (struct libjpeg::jpeg_common_struct *cinfo)
{
    ErrorContext &errCtx = *(ErrorContext*)cinfo->err;
    NN_LOG("[libjpeg returned with error %d(%d)]\n", errCtx.errMgr.msg_code, errCtx.errMgr.msg_parm.i[0]);

    // エラー値が期待通りかを検査。
    errCtx.isExpected = errCtx.handler(errCtx.errMgr.msg_code, errCtx.errMgr.msg_parm.i[0]);

    // setjmp した場所に戻る。例外が使えないためjmpで。
    longjmp(errCtx.jmpBuf, -1);
}
}

bool DecompressBinary(
    uint32_t *outPixel,
    const size_t kPixelSize,
    const void *jpeg,
    const size_t kJpegSize,
    const TestArgDecode &decodeArg,
    void *temp,
    const size_t kTempSize)
{
    volatile bool isOk = false;
    uint8_t *lines[LibjpegProperty_NumLinesToProcAtOnce] = {};
    for (int i = 0; i < LibjpegProperty_NumLinesToProcAtOnce; i++)
    {
        lines[i] = nullptr;
    }

    libjpeg::jpeg_decompress_struct dinfo; // Libjpeg の内部状態管理用構造体

    // Libjpeg の内部状管理用構造体を設定
    libjpeg::jpeg_workbuf workBuf = {};
    workBuf.ptr = temp;
    workBuf.total = kTempSize;
    dinfo.workbuf = &workBuf;

    ErrorContext errCtx = {};
    dinfo.err = libjpeg::jpeg_std_error(&errCtx.errMgr);
    errCtx.errMgr.error_exit = ErrorExit;
    errCtx.handler = decodeArg.errorHandler;
    errCtx.isExpected = false;


    // Libjpeg の内部状態を初期化
    libjpeg::jpeg_create_decompress(&dinfo);

    // Exception handling
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
    if (setjmp(errCtx.jmpBuf) != 0)
    {
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        // エラー時は「非0が返った」というコンテクストになる。
        isOk = errCtx.isExpected;
        goto out;
    }

    do
    {
        libjpeg::jpeg_mem_src(
            &dinfo,
            const_cast<uint8_t*>(reinterpret_cast<const uint8_t*>(jpeg)),
            static_cast<unsigned long>(kJpegSize));

        // JPEG ヘッダ解析
        if (libjpeg::jpeg_read_header(&dinfo, false) != JPEG_HEADER_OK)
        {
            errCtx.errMgr.msg_code = libjpeg::JERR_CANT_SUSPEND;
            errCtx.errMgr.error_exit((libjpeg::j_common_ptr)&dinfo);
        }

        // Decode 設定
        dinfo.scale_num = 1;
        dinfo.scale_denom = decodeArg.resolutionDenom;
        dinfo.out_color_space = libjpeg::JCS_RGB;
        dinfo.dct_method = libjpeg::JDCT_ISLOW;
        dinfo.do_fancy_upsampling = FALSE;
        dinfo.do_block_smoothing = FALSE;

        // 中間変数のセットアップ
        if (!libjpeg::jpeg_start_decompress(&dinfo))
        {
            NN_LOG("[jpeg_start_decompress usually does not fail]\n");
            break;
        }

        // 出力サイズが与えられた出力バッファを上回らないか？
        if (dinfo.output_width * dinfo.output_height * sizeof(uint32_t) > kPixelSize)
        {
            NN_LOG(
                "[Decoded pixel data is too large: %u (%dx%d)]\n",
                dinfo.output_width * dinfo.output_height * sizeof(uint32_t),
                dinfo.output_width, dinfo.output_height);
            break;
        }

        // 出力サイズが期待値通りか？
        NN_LOG(
            "[JPEG data: %dx%dx%d]\n",
            dinfo.output_width, dinfo.output_height, dinfo.output_components);
        if (!(dinfo.output_width == decodeArg.expectedDim.width &&
            dinfo.output_height == decodeArg.expectedDim.height))
        {
            EXPECT_EQ(decodeArg.expectedDim.width, dinfo.output_width);
            EXPECT_EQ(decodeArg.expectedDim.height, dinfo.output_height);
            break;
        }
        // 出力色数は3か？
        if (dinfo.output_components != 3)
        {
            NN_LOG("[dinfo.output_components must be 3]\n");
            break;
        }

        // 一時出力用バッファの確保。テストでは何も考えずmallocする。
        int rowStride = dinfo.output_components * dinfo.output_width;
        for (int i = 0; i < LibjpegProperty_NumLinesToProcAtOnce; i++)
        {
            lines[i] = (uint8_t*)malloc(rowStride);
            if (lines[i] == nullptr)
            {
                NN_LOG("[Allocation error]\n");
                break;
            }
        }

        /* Decode */
        while (dinfo.output_scanline < dinfo.output_height)
        {
            int currentLine = dinfo.output_scanline;
            int linesToDecode = dinfo.output_height - currentLine;
            linesToDecode = (linesToDecode < LibjpegProperty_NumLinesToProcAtOnce)? linesToDecode: LibjpegProperty_NumLinesToProcAtOnce;

            int readCt = libjpeg::jpeg_read_scanlines(&dinfo, lines, linesToDecode);
            for (int i = 0; i < readCt; i++)
            {
                uint32_t *pixelLine = outPixel + (currentLine + i) * dinfo.output_width;
                libjpeg::JSAMPLE *line = lines[i];

                for (int32_t j = 0; j < dinfo.output_width; j++, line += 3)
                {
                    // Copy 3*8bits from decoded line to upper 24bit of 32bit texel.
                    pixelLine[j] = (line[0] << 24) + (line[1] << 16) + (line[2] << 8) + 0xFF;
                }
            }
        }

        // デコードの後始末。別にやらなくてもいいが一応。
        if (!libjpeg::jpeg_finish_decompress(&dinfo))
        {
            NN_LOG("[jpeg_start_decompress usually does not fail]\n");
            break;
        }

        isOk = decodeArg.errorHandler(0, 0);
    } while (NN_STATIC_CONDITION(false));

out:
    // 内部状態管理用構造体の後始末。別にやらなくてもいいが一応。
    libjpeg::jpeg_destroy_decompress(&dinfo);
    for (int i = 0; i < LibjpegProperty_NumLinesToProcAtOnce; i++)
    {
        if (lines[i] != nullptr)
        {
            free(lines[i]);
        }
    }

    return isOk;
} // NOLINT(readability/fn_size)

#define ROUND_UP(v, d) (((v) % (d)) != 0? ((v) - ((v) % (d)) + (d)): (v))

bool CompressBinary(
    size_t *pOutActualCodedSize,
    void *outJpegBuf,
    const size_t kJpegBufSize,
    const uint32_t *pixel,
    const Dimension &kDim,
    const TestArgEncode &encodeArg,
    void *workBuf,
    const size_t kWorkBufSize)
{
    const int kRowStride = kDim.width * LibjpegProperty_BytesPerPixel;

    volatile bool isOk = false;
    uint8_t *lines[LibjpegProperty_NumLinesToProcAtOnce] = {};
    for (int i = 0; i < LibjpegProperty_NumLinesToProcAtOnce; i++)
    {
        lines[i] = (uint8_t*)malloc(kRowStride);
        if (lines[i] == nullptr)
        {
            return false;
        }
    }

    libjpeg::jpeg_workbuf wbMgr = {};
    wbMgr.ptr = (reinterpret_cast<unsigned char*>(workBuf));
    wbMgr.total = kWorkBufSize;

    libjpeg::jpeg_compress_struct cinfo = {};
    cinfo.workbuf = &wbMgr;

    ErrorContext errCtx = {};
    cinfo.err = libjpeg::jpeg_std_error(&errCtx.errMgr);
    errCtx.errMgr.error_exit = ErrorExit;
    errCtx.handler = encodeArg.errorHandler;
    errCtx.isExpected = false;

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
    if (setjmp(errCtx.jmpBuf) != 0)
    {
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        EXPECT_TRUE(errCtx.isExpected);
        goto out;
    }

    do
    {
        // Prepare for encoding
        libjpeg::jpeg_create_compress(&cinfo);

        unsigned long codedSize = static_cast<decltype(codedSize)>(kJpegBufSize);
        libjpeg::jpeg_mem_dest(&cinfo, reinterpret_cast<unsigned char**>(&outJpegBuf), &codedSize);

        cinfo.image_width       = kDim.width;
        cinfo.image_height      = kDim.height;
        cinfo.input_components  = LibjpegProperty_BytesPerPixel;
        cinfo.in_color_space    = libjpeg::JCS_RGB;

        // Set encoding parameters
        // Base encoding setting.
        libjpeg::jpeg_set_defaults(&cinfo);
        libjpeg::jpeg_set_colorspace(&cinfo, libjpeg::JCS_YCbCr);
        libjpeg::jpeg_set_quality(&cinfo, encodeArg.quality, true);
        cinfo.comp_info[0].h_samp_factor = encodeArg.yHSample;
        cinfo.comp_info[0].v_samp_factor = encodeArg.yVSample;

        // Performance configuration for fast encoding.
        cinfo.dct_method = libjpeg::JDCT_ISLOW;
        cinfo.optimize_coding = false;
        cinfo.do_fancy_downsampling = false;

        // Start encoding
        libjpeg::jpeg_start_compress(&cinfo, true);

        // Encoding process
        while (cinfo.next_scanline < kDim.height)
        {
            int currentLine = cinfo.next_scanline;
            int linesToEncode = kDim.height - currentLine;
            linesToEncode = (linesToEncode < LibjpegProperty_NumLinesToProcAtOnce)? linesToEncode: LibjpegProperty_NumLinesToProcAtOnce;

            // Convert 32-bit texture into 24-bit texture
            for (int i = 0; i < linesToEncode; i++)
            {
                const uint32_t *pixelLine = pixel + (currentLine + i) * kDim.width;
                libjpeg::JSAMPLE *line = lines[i];

                for (int j = 0; j < kDim.width; j++, line += LibjpegProperty_BytesPerPixel)
                {
                    uint32_t px = pixelLine[j];
                    line[0] = (libjpeg::JSAMPLE)((px & 0xFF000000) >> 24);
                    line[1] = (libjpeg::JSAMPLE)((px & 0x00FF0000) >> 16);
                    line[2] = (libjpeg::JSAMPLE)((px & 0x0000FF00) >> 8);
                }
            }

            // Encode to JPEG from 24-bit texture
            libjpeg::jpeg_write_scanlines(&cinfo, lines, linesToEncode);
        }

        libjpeg::jpeg_finish_compress(&cinfo);
        *pOutActualCodedSize = codedSize;

        isOk = encodeArg.errorHandler(0, 0);
    } while (NN_STATIC_CONDITION(false));


out:
    libjpeg::jpeg_destroy_compress(&cinfo);

    for (int i = 0; i < LibjpegProperty_NumLinesToProcAtOnce; i++)
    {
        if (lines[i] != nullptr)
        {
            free(lines[i]);
        }
    }

    return isOk || errCtx.isExpected;
} // NOLINT(readability/fn_size)

}}}
