﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/i2c/i2c.h>
#include <nn/init.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/result/result_HandlingUtility.h>

#include <eval_Screen.h>

namespace {

const nn::i2c::I2cDevice I2cDeviceName = nn::i2c::I2cDevice::I2cDevice_Bq24193;
nn::i2c::I2cSession g_I2cSession;

nnt::eval::Screen* g_pScreen;
nn::gfx::util::DebugFontTextWriter* g_DebugFontTextWriter;

const int InformationBufferLength = 128;
char g_InformationBuffer[InformationBufferLength];
int g_InformationBufferOffset = 0;

int g_FuelGauge = 0;
const int FuelGaugeThreshold = 100;
nn::psm::ChargerType g_ChargerType;

void DrawScreen() NN_NOEXCEPT
{
    g_pScreen->Draw();
}

void DrawDisplay() NN_NOEXCEPT
{
    // Defaultとするフォントサイズ
    const float FontScale = 2.5f;

    g_DebugFontTextWriter->SetScale(FontScale, FontScale);
    g_DebugFontTextWriter->SetCursor(60.0f, 60.0f);
    g_DebugFontTextWriter->Print("%s\n", g_InformationBuffer);
    g_DebugFontTextWriter->Print("FUEL:%3d%%\n", g_FuelGauge);

    if ( g_FuelGauge >= FuelGaugeThreshold )
    {
        g_DebugFontTextWriter->Print("Finished!!!\n");
        nnt::eval::SetBackGroundColor(nnt::eval::Green);
    }
    else if ( g_ChargerType == nn::psm::ChargerType_EnoughPower )
    {
        nnt::eval::SetBackGroundColor(nnt::eval::Black);
    }
    else
    {
        nnt::eval::SetBackGroundColor(nnt::eval::Red);
    }

    DrawScreen();
}

void UpdateICharge() NN_NOEXCEPT
{
    const int     ShiftICharge = 2;
    const uint8_t MaskICharge = 0xfc;
    const uint8_t ICharge2A = 0b011000;
    const uint8_t Address = 0x02;
    uint8_t reg;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, g_I2cSession, &Address));

    NN_LOG("REG02:%02x\n", reg);

    if ( (reg & MaskICharge) != (ICharge2A << ShiftICharge) )
    {
        NN_LOG("Not 2A!\n");

        reg = (reg & (~MaskICharge)) | ((ICharge2A << ShiftICharge) & MaskICharge);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(g_I2cSession, &Address, &reg));

        reg = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, g_I2cSession, &Address));
        NN_LOG("Overwrite REG02:%02x\n", reg);
    }
}

void RestoreICharge() NN_NOEXCEPT
{
    const int     ShiftICharge = 2;
    const uint8_t MaskICharge = 0xfc;
    const uint8_t ICharge512A = 0b000000;
    const uint8_t Address = 0x02;
    uint8_t reg;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, g_I2cSession, &Address));

    reg = (reg & (~MaskICharge)) | ((ICharge512A << ShiftICharge) & MaskICharge);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(g_I2cSession, &Address, &reg));

    reg = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, g_I2cSession, &Address));
    NN_LOG("Restore REG02:%02x\n", reg);
}

void UpdateFuelGauge() NN_NOEXCEPT
{
    g_FuelGauge = nn::psm::GetBatteryChargePercentage();
    g_ChargerType = nn::psm::GetChargerType();
}

void ClearInformation() NN_NOEXCEPT
{
    g_InformationBufferOffset = 0;
    g_InformationBuffer[0] = '\0';
}

#define WRITE_INFORMATION(...) \
{ \
    int count = std::snprintf(&g_InformationBuffer[g_InformationBufferOffset], InformationBufferLength - g_InformationBufferOffset, __VA_ARGS__); \
    if ( count > 0 ) \
    { \
        g_InformationBufferOffset += count; \
    } \
}

void UpdateInformation() NN_NOEXCEPT
{
    static int count = 0;

    if ( g_ChargerType == nn::psm::ChargerType_EnoughPower )
    {
        WRITE_INFORMATION("Now Charging");
        for ( int i = 0; i < count; i++ )
        {
            WRITE_INFORMATION(".");
        }
    }
    else
    {
        WRITE_INFORMATION("Please connect an AC adaptor or a cradle.");
    }
    count++;
    count %= 4;
}

void Update() NN_NOEXCEPT
{
    // 現在の Icharge の取得と設定更新。
    UpdateICharge();

    // 電池残量の取得。
    UpdateFuelGauge();

    // Information Buffer の更新。
    UpdateInformation();
}

} // namespace

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    const size_t MallocHeapSize = 256 * 1024 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    ::nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

extern "C" void nnMain()
{
    // OE の設定による性能の低下。
    nn::oe::Initialize();
    nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz);

    g_pScreen = new ::nnt::eval::Screen();
    g_pScreen->Initialize();
    g_DebugFontTextWriter = g_pScreen->GetDebugFontTextWriterPtr();

    nn::i2c::Initialize();
    nn::i2c::OpenSession(&g_I2cSession, I2cDeviceName);

    nn::os::SystemEventType sleepButtonEvent;
    nn::os::CreateSystemEvent(&sleepButtonEvent, nn::os::EventClearMode_ManualClear, true);

    nn::hid::system::InitializeSleepButton();
    nn::hid::system::BindSleepButtonEvent(&sleepButtonEvent, nn::os::EventClearMode_ManualClear);

    nn::os::ClearSystemEvent(&sleepButtonEvent);

    nn::psm::Initialize();

    // SleepButton 長押し維持ループ数。
    int shutdownCount = 0;
    bool active = false;

    while ( true )
    {
        // 雑に Information Buffer を空にする。
        ClearInformation();

        bool event = nn::os::TimedWaitSystemEvent(&sleepButtonEvent, nn::TimeSpan::FromMilliSeconds(1000LL));

        if ( event )
        {
            nn::os::ClearSystemEvent(&sleepButtonEvent);

            nn::hid::system::SleepButtonState sleepButtonState;
            nn::hid::system::GetSleepButtonStates(&sleepButtonState, 1);
            if ( sleepButtonState.buttons.Test<nn::hid::system::SleepButton::Active>() )
            {
                WRITE_INFORMATION("Power button pushed.");
                active = true;
            }
            else
            {
                WRITE_INFORMATION("Power button released.");
                active = false;
            }
            shutdownCount = 0;
        }
        else
        {
            if ( active )
            {
                WRITE_INFORMATION("Shutdown count down %d.", 2 - shutdownCount);

                if ( shutdownCount >= 2 )
                {
                    break;
                }
                else
                {
                    shutdownCount++;
                }
            }
            else
            {
                Update();
            }
        }

        // 画面の描画。（ICharge、電池残量、ボタン入力と処理、ハングしてないことを示すアイコン）
        DrawDisplay();
    }

    RestoreICharge();

    nn::psm::Finalize();

    nn::os::DestroySystemEvent(&sleepButtonEvent);

    nn::i2c::CloseSession(g_I2cSession);
    nn::i2c::Finalize();

    g_pScreen->Finalize();
    delete g_pScreen;

    nn::bpc::InitializeBoardPowerControl();
    nn::bpc::ShutdownSystem();
}
