﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nnt/nntest.h>

#include <nn/nn_Windows.h>
#include <nn/hws.h>

#define NN_TESTWND_EXPECT_VALID(handle) \
    EXPECT_TRUE((handle).IsValid())

#define NN_TESTWND_EXPECT_INVALID(handle) \
    EXPECT_FALSE((handle).IsValid())

#define NN_TESTWND_ASSERT_VALID(handle) \
    ASSERT_TRUE((handle).IsValid())

namespace {

std::unique_ptr<char[]> GetWindowTitle( nn::hws::WindowHandle handle )
{
    HWND hWnd = static_cast<HWND>( handle.Get() );
    int lenTitle = GetWindowTextLengthA( hWnd ) + 1;
    std::unique_ptr<char[]> title( new char[ lenTitle ] );
    GetWindowTextA( hWnd, title.get(), lenTitle );
    return std::move( title );
}

class TestWindow
{
public:
    TestWindow()
    {
    }
    TestWindow( const char* windowName, const char* title )
    {
        m_Arg.SetWindowName( windowName );
        m_Arg.SetTitle( title );
    }

    virtual ~TestWindow()
    {
        Destroy();
    }

    nn::hws::CreateWindowArg& GetCreateArg()
    {
        return m_Arg;
    }
    nn::hws::WindowHandle GetHandle()
    {
        return m_Handle;
    }

    HWND GetHWnd()
    {
        return static_cast<HWND>( m_Handle.Get() );
    }

    nn::hws::WindowHandle Create()
    {
        return ( m_Handle = nn::hws::CreateWindow( m_Arg ) );
    }

    void Destroy()
    {
        if( m_Handle.IsValid() )
        {
            nn::hws::DestroyWindow( m_Handle );
            m_Handle.Invalidate();
        }
    }
private:
    nn::hws::WindowHandle m_Handle;
    nn::hws::CreateWindowArg m_Arg;
};

}

TEST( BasicTest, WindowHandle )
{
    nn::hws::WindowHandle handle;
    NN_TESTWND_EXPECT_INVALID( handle );
    handle.Invalidate();
    NN_TESTWND_EXPECT_INVALID( handle );
}

TEST( BasicTest, WindowCreateArg )
{
    const char* windowName = "Window0";
    const char* title = "Title0";
    int left = 100;
    int top = 50;
    int width = 640;
    int height = 480;

    nn::hws::CreateWindowArg arg;
    arg.SetWindowName( windowName );
    arg.SetTitle( title );
    arg.SetLeft( left );
    arg.SetTop( top );
    arg.SetWidth( width );
    arg.SetHeight( height );

    EXPECT_EQ( windowName, arg.GetWindowName() );
    EXPECT_EQ( title, arg.GetTitle() );
    EXPECT_EQ( left, arg.GetLeft() );
    EXPECT_EQ( top, arg.GetTop() );
    EXPECT_EQ( height, arg.GetHeight() );
    EXPECT_STREQ( windowName, arg.GetWindowName() );
    EXPECT_STREQ( title, arg.GetTitle() );
}

TEST( BasicTest, Create )
{
    TestWindow window;
    {
        nn::hws::CreateWindowArg& arg = window.GetCreateArg();
        arg.SetWindowName( "Window0" );
        arg.SetTitle( "Title0" );
        arg.SetLeft( 40 );
        arg.SetTop( 80 );
        arg.SetWidth( 640 );
        arg.SetHeight( 480 );
        NN_TESTWND_ASSERT_VALID( window.Create() );
    }

    {
        const int bufSize = 256;
        char buf[ bufSize ];
        ASSERT_TRUE( GetClassNameA( window.GetHWnd(), buf, bufSize ) > 0 );
        EXPECT_STREQ( window.GetCreateArg().GetWindowName(), buf );

        EXPECT_STREQ( window.GetCreateArg().GetTitle(), GetWindowTitle( window.GetHandle() ).get() );

        RECT windowRect;
        ASSERT_EQ( TRUE, GetWindowRect( window.GetHWnd(), &windowRect ) );
        EXPECT_EQ( window.GetCreateArg().GetLeft(), windowRect.left );
        EXPECT_EQ( window.GetCreateArg().GetTop(), windowRect.top );

        RECT clientRect;
        ASSERT_EQ( TRUE, GetClientRect( window.GetHWnd(), &clientRect ) );
        EXPECT_EQ( window.GetCreateArg().GetWidth(), clientRect.right );
        EXPECT_EQ( window.GetCreateArg().GetHeight(), clientRect.bottom );
    }
}

TEST( BasicTest, Destroy )
{
    TestWindow window( "Window0", "Title0" );
    NN_TESTWND_ASSERT_VALID( window.Create() );
    EXPECT_EQ( TRUE, IsWindow( window.GetHWnd() ) );
    window.Destroy();
    EXPECT_EQ( FALSE, IsWindow( window.GetHWnd() ) );
}

TEST( BasicTest, FindWindow )
{
    TestWindow window0( "Window0", "Title0" );
    TestWindow window1( "Window1", "Title1" );
    NN_TESTWND_ASSERT_VALID( window0.Create() );
    NN_TESTWND_ASSERT_VALID( window1.Create() );
    {
        nn::hws::WindowHandle found0 = nn::hws::FindWindow( window0.GetCreateArg().GetWindowName() );
        nn::hws::WindowHandle found1 = nn::hws::FindWindow( window1.GetCreateArg().GetWindowName() );
        NN_TESTWND_EXPECT_INVALID( nn::hws::FindWindow( "Window2" ) );
        NN_TESTWND_ASSERT_VALID( found0 );
        NN_TESTWND_ASSERT_VALID( found1 );
        EXPECT_NE( found0.Get(), found1.Get() );
        EXPECT_STREQ( window0.GetCreateArg().GetTitle(), GetWindowTitle( found0 ).get() );
        EXPECT_STREQ( window1.GetCreateArg().GetTitle(), GetWindowTitle( found1 ).get() );
    }
    window0.Destroy();
    {
        NN_TESTWND_EXPECT_INVALID( nn::hws::FindWindow( window0.GetCreateArg().GetWindowName() ) );
        nn::hws::WindowHandle found1 = nn::hws::FindWindow( window1.GetCreateArg().GetWindowName() );
        NN_TESTWND_ASSERT_VALID( found1 );
        EXPECT_STREQ( window1.GetCreateArg().GetTitle(), GetWindowTitle( found1 ).get() );
    }
    window1.Destroy();
    {
        NN_TESTWND_EXPECT_INVALID( nn::hws::FindWindow( window0.GetCreateArg().GetWindowName() ) );
        NN_TESTWND_EXPECT_INVALID( nn::hws::FindWindow( window1.GetCreateArg().GetWindowName() ) );
    }
}

TEST( BasicTest, ShowWindow )
{
    TestWindow window( "Window", "Title" );
    NN_TESTWND_ASSERT_VALID( window.Create() );
    EXPECT_EQ( FALSE, IsWindowVisible( window.GetHWnd() ) );
    nn::hws::ShowWindow( window.GetHandle() );
    EXPECT_EQ( TRUE, IsWindowVisible( window.GetHWnd() ) );
}

TEST( BasicTest, MouseWheel )
{
    TestWindow window( "Window", "Title" );
    NN_TESTWND_ASSERT_VALID( window.Create() );
    nn::hws::ProcessMessage();
    EXPECT_EQ( nn::hws::GetMouseWheel(), 0 );
    SendMessageA( window.GetHWnd(), WM_MOUSEWHEEL,
        static_cast<WPARAM>( WHEEL_DELTA << 16 | MK_RBUTTON ),
        static_cast<LPARAM>( ( 10 << 16 ) | 20 ) );
    EXPECT_EQ( nn::hws::GetMouseWheel(), WHEEL_DELTA );
    nn::hws::ProcessMessage();
    EXPECT_EQ( nn::hws::GetMouseWheel(), 0 );
    SendMessageA( window.GetHWnd(), WM_MOUSEWHEEL,
        static_cast<WPARAM>( ( WHEEL_DELTA * -4 ) << 16 | MK_XBUTTON1),
        static_cast<LPARAM>( ( 100 << 16 ) | 0 ) );
    EXPECT_EQ( nn::hws::GetMouseWheel(), WHEEL_DELTA * -4 );
    SendMessageA( window.GetHWnd(), WM_MOUSEWHEEL,
        static_cast<WPARAM>( ( WHEEL_DELTA * 2 ) << 16 ), 0 );
    EXPECT_EQ( nn::hws::GetMouseWheel(), WHEEL_DELTA * ( -4 + 2 ) );
    nn::hws::ProcessMessage();
    EXPECT_EQ( nn::hws::GetMouseWheel(), 0 );
}

TEST( BasicTest, ProcessMessage )
{
    MSG msg;
    TestWindow window( "Window", "Title" );
    NN_TESTWND_ASSERT_VALID( window.Create() );
    nn::hws::ProcessMessage();
    EXPECT_EQ( TRUE, CloseWindow( window.GetHWnd() ) );
    nn::hws::ProcessMessage();
    EXPECT_FALSE( PeekMessage( &msg, nullptr, 0, 0, PM_NOREMOVE ) );
    SendMessageA( window.GetHWnd(), WM_CLOSE, 0, 0 );
    EXPECT_EXIT( nn::hws::ProcessMessage(), testing::ExitedWithCode( 0 ), "" );
}
