﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace StressTestTool
{
    internal class Program
    {
        private static HtcsCommunicator m_HtcsCommunicator = new HtcsCommunicator();

        private struct TestParams
        {
            public string keyString;
            public string testName;
            public ManualResetEvent finishedEvent;
        };

        private static void TestManager(StressTestToolParams args)
        {
            var tcpListener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            tcpListener.Start();

            var htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "HtcsTestManager");
            var portMapItem = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)tcpListener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapItem) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            WaitCallback waitCallback = new WaitCallback(TestBody);

            var finishedEvents = new ManualResetEvent[args.Count];
            for (int i = 0; i < args.Count; i++)
            {
                var client = tcpListener.AcceptTcpClient();

                TestParams param;

                using (var reader = new StreamReader(client.GetStream(), Encoding.ASCII, true))
                {
                    var data = reader.ReadToEnd().Trim('\0').Split(',');
                    param.keyString = data[0];
                    param.testName = data[1];

                    finishedEvents[i] = new ManualResetEvent(false);
                    param.finishedEvent = finishedEvents[i];
                }

                client.Close();

                ThreadPool.QueueUserWorkItem(waitCallback, param);
            }

            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);

            tcpListener.Stop();

            Console.WriteLine("Waiting all tests end.");
            WaitHandle.WaitAll(finishedEvents);
            Console.WriteLine("All tests finished.");
        }

        private static void TestBody(object obj)
        {
            TestParams testParams = (TestParams)obj;
            Console.WriteLine("Start " + testParams.testName);
            switch (testParams.testName)
            {
                case "TypicalCase":
                    {
                        // Repeat
                        TestServer server0 = new TestServer(0, "HostToTarget_" + testParams.keyString, m_HtcsCommunicator, (client) => Utils.Send(client, 1024, 1024));
                        server0.Start();

                        TestServer server1 = new TestServer(0, "TargetToHost_" + testParams.keyString, m_HtcsCommunicator, (client) => Utils.Recv(client, 1024, 1024));
                        server1.Start();

                        server0.Wait();
                        server1.Wait();

                        // MultiPort
                        const int ThreadCount = 8;
                        TestServer[] servers = new TestServer[ThreadCount];

                        foreach(int i in Enumerable.Range(0, ThreadCount))
                        {
                            servers[i] = new TestServer(0, "Echo" + i + "_" + testParams.keyString, m_HtcsCommunicator, (client) => Utils.Echo(client, 1024 * 1024, 128));
                            servers[i].Start();
                        }

                        foreach (int i in Enumerable.Range(0, ThreadCount))
                        {
                            servers[i].Wait();
                        }
                    }
                    break;
                case "BigData":
                    {
                        const int EchoDataSize = 512 * 1024 * 1024;

                        TestServer echoServer = new TestServer(0, "Echo0_" + testParams.keyString, m_HtcsCommunicator, (client) => Utils.Echo(client, EchoDataSize));
                        echoServer.Start();
                        echoServer.Wait();

                        TestClient echoClient = new TestClient("Echo1_" + testParams.keyString, m_HtcsCommunicator, (client) => Utils.Echo(client, EchoDataSize));
                        echoClient.Start();
                        echoClient.Wait();
                    }
                    break;
                case "SpecialCase":
                    {
                        var test = new SpecialCaseTest(testParams.keyString, m_HtcsCommunicator);
                        test.Start();
                    }
                    break;
                case "VariousSize":
                    {
                        const int MaxDataSize = 16 * 1024;

                        TestServer echoServer = new TestServer(0, "Echo0_" + testParams.keyString, m_HtcsCommunicator, (client) => {
                            for (int size = 1; size <= MaxDataSize; size++)
                            {
                                Utils.Echo(client, size);
                            }
                        });
                        echoServer.Start();
                        echoServer.Wait();

                        TestClient echoClient = new TestClient("Echo1_" + testParams.keyString, m_HtcsCommunicator, (client) => {
                            for (int size = 1; size <= MaxDataSize; size++)
                            {
                                Utils.Echo(client, size);
                            }
                        });
                        echoClient.Start();
                        echoClient.Wait();
                    }
                    break;
                case "ConnectTest":
                    {
                        var test = new ConnectTest(testParams.keyString, m_HtcsCommunicator);
                        test.Start();
                    }
                    break;
                case "DisconnectTest":
                    {
                        var test = new DisconnectTest(testParams.keyString, m_HtcsCommunicator);
                        test.Start();
                    }
                    break;
                case "NonBlockingTest":
                    {
                        var test = new NonBlockingTest(testParams.keyString, m_HtcsCommunicator);
                        test.Start();
                    }
                    break;
                case "Stress":
                    {
                        var test = new StressTest(testParams.keyString, m_HtcsCommunicator);
                        test.Start();
                    }
                    break;
                default:
                    Console.WriteLine("No such test case: {0}", testParams.testName);
                    break;
            }

            testParams.finishedEvent.Set();
        }

        private static void PrintHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("-  TargetList -");
            foreach (Target item in e.TargetList)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");

            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem item in e.PortMap)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
        }

        private static void Main(string[] args)
        {
            StressTestToolArgs opt = new StressTestToolArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            // HtcsInfo が更新されたときに呼ばれる関数を登録
            //   HtcsInfo リスト表示関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += PrintHtcsInfo;

            // TargetManager への接続が確立されるまでループ
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.WriteLine("Failed connecting to TargetManager.");
                Thread.Sleep(1000);
            }

            TestManager(opt.Params);

            m_HtcsCommunicator.Stop();
        }
    }
}
