﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace EchoServer
{
    internal class Program
    {
        private static HtcsCommunicator m_HtcsCommunicator = new HtcsCommunicator();
        private static TargetList m_TargetList;
        private static PortMap m_PortMap;
        private static bool m_EndFlag = false;

        private static void ServerSample()
        {
            // サーバの構築
            const int PortNumber = 23445;
            TcpListener listener = new TcpListener(System.Net.IPAddress.Loopback, PortNumber);
            Thread thread = new Thread(ServerFunc);
            thread.Start(listener);

            // TargetManager にサーバ情報を登録
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(HtcsPeerName.Any, "ServerInHost");
            PortMapItem portMapping = new PortMapItem(htcsPortDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error.");
                Thread.Sleep(1000);
            }

            thread.Join();

            // 登録していたポート情報を削除します。
            m_HtcsCommunicator.UnregisterPort(htcsPortDesc);

            listener.Stop();
            Console.WriteLine("Closed.");
        }

        private static void ServerFunc(object obj)
        {
            TcpListener listener = ((TcpListener)obj);
            listener.Start(10);

            while (!m_EndFlag)
            {
                Console.WriteLine("Waiting for connection from Target.");
                TcpClient client = listener.AcceptTcpClient();
                Console.WriteLine("Accepted client on target.");

                EchoBack(client);
            }
        }

        // 送られてきたデータをそのまま相手に送り返す
        private static void EchoBack(TcpClient client)
        {
            while (true)
            {
                string read;
                using (var sr = new StreamReader(client.GetStream(), Encoding.ASCII, true, 1024, true))
                {
                    read = sr.ReadLine();

                    if (read == null) // ソケットがクローズされたら抜ける
                    {
                        break;
                    }
                    else if (read.Equals("End")) // 終了メッセージを受信したら終了フラグを立てて抜ける
                    {
                        m_EndFlag = true;
                        break;
                    }
                }
                Console.WriteLine("Received \"{0}\".", read);
                using (var sw = new StreamWriter(client.GetStream(), Encoding.ASCII, 1024, true))
                {
                    sw.Write(read);
                    sw.Flush();
                }
            }
        }

        private static void CheckHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            // HtcsInfo のキャッシュを更新
            m_TargetList = e.TargetList;
            m_PortMap = e.PortMap;
        }

        private static void PrintHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("-  TargetList -");
            foreach (Target item in e.TargetList)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem item in e.PortMap)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
        }

        private static void Main(string[] args)
        {
            // HtcsInfo が更新されたときに呼ばれる関数を登録
            //   HtcsInfo リスト表示関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += PrintHtcsInfo;
            //   HtcsInfo 内のポートマッピングに、接続予定のポートが追加されたかをチェックする関数の登録
            m_HtcsCommunicator.HtcsInfoUpdated += CheckHtcsInfo;

            // TargetManager への接続が確立されるまでループ
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.WriteLine("Failed connecting to TargetManager.");
                Console.WriteLine("Press Enter to retry.");
                Console.ReadLine();
            }

            ServerSample();

            m_HtcsCommunicator.Stop();

            Console.WriteLine("Sample end");
            Console.ReadLine();
        }
    }
}
