﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <map>
#include <nn/nn_Log.h>
#include <nn/htcs.h>
#include <nn/os.h>
#include <nnt.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

/*
*   ベンチマークテストの Client
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/BulkServer を実行する必要があります。
*/

namespace
{
    enum ThresholdType
    {
        ThresholdType_Default,
        ThresholdType_SdevWithJumboFrame,
        ThresholdType_SdevWithoutJumboFrame,
        ThresholdType_Edev,
    } g_Type;

    // Deafult は一番閾値の緩いものに合わせる
    std::map<ThresholdType, const char*> StrThresholdType =
    {
        { ThresholdType_Default , "Default" },
        { ThresholdType_SdevWithJumboFrame , "SdevWithJumboFrame" },
        { ThresholdType_SdevWithoutJumboFrame ,  "SdevWithoutJumboFrame" },
        { ThresholdType_Edev , "Edev" },
    };
    std::map<ThresholdType, double> LatencyThreshold =
    {
        { ThresholdType_Default , 0.0025 },
        { ThresholdType_SdevWithJumboFrame , 0.0018 },
        { ThresholdType_SdevWithoutJumboFrame ,  0.0015 },
        { ThresholdType_Edev , 0.0025 },
    };
    std::map<ThresholdType, double> T2HBandWidthThreshold =
    {
        { ThresholdType_Default , 15.0 * 1024 * 1024 },
        { ThresholdType_SdevWithJumboFrame , 30.0 * 1024 * 1024 },
        { ThresholdType_SdevWithoutJumboFrame ,  20.0 * 1024 * 1024 },
        { ThresholdType_Edev , 15.0 * 1024 * 1024 },
    };
    std::map<ThresholdType, double> H2TBandWidthThreshold =
    {
        { ThresholdType_Default , 15.0 * 1024 * 1024 },
        { ThresholdType_SdevWithJumboFrame , 30.0 * 1024 * 1024 },
        { ThresholdType_SdevWithoutJumboFrame ,  15.0 * 1024 * 1024 },
        { ThresholdType_Edev , 15.0 * 1024 * 1024 },
    };

    void ParseArguments(int argc, char** argv)
    {
        if (argc < 2)
        {
            g_Type = ThresholdType_Default;
            return;
        }

        const char strSdevWithJumbo[] = "type=SdevWithJumboFrame";
        const char strSdevWithoutJumbo[] = "type=SdevWithoutJumboFrame";
        const char strEdev[] = "type=Edev";
        if (strncmp(argv[1], strSdevWithJumbo, sizeof(strSdevWithJumbo)) == 0)
        {
            g_Type = ThresholdType_SdevWithJumboFrame;
        }
        else if (strncmp(argv[1], strSdevWithoutJumbo, sizeof(strSdevWithoutJumbo)) == 0)
        {
            g_Type = ThresholdType_SdevWithoutJumboFrame;
        }
        else if (strncmp(argv[1], strEdev, sizeof(strEdev)) == 0)
        {
            g_Type = ThresholdType_Edev;
        }
        else
        {
            g_Type = ThresholdType_Default;
        }
    }

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class BenchmarkClient : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    // 指定されたサイズのデータを送受信して、かかった時間（秒）を返す
    double BulkCommunication(int socket, uint8_t sendBuffer[], const int32_t sendByteCount, uint8_t recvBuffer[], const int32_t recvByteCount)
    {
        nn::htcs::ssize_t size;

        // 送受信サイズを送信（ホストと同一エンディアンであると仮定）
        size = nn::htcs::Send(socket, &sendByteCount, sizeof(int32_t), 0);
        EXPECT_EQ(sizeof(int32_t), size);
        if (size < 0)
        {
            NN_LOG("L%d failed(%d)\n", __LINE__, nn::htcs::GetLastError());
        }
        size = nn::htcs::Send(socket, &recvByteCount, sizeof(int32_t), 0);
        EXPECT_EQ(sizeof(int32_t), size);
        if (size < 0)
        {
            NN_LOG("L%d failed(%d)\n", __LINE__, nn::htcs::GetLastError());
        }

        // 送受信サイズが両方0なら通信終了
        if (sendByteCount == 0 && recvByteCount == 0)
        {
            return 0.0;
        }

        // Ack 受信
        uint8_t ack;
        size = nn::htcs::Recv(socket, &ack, 1, 0);

        nn::os::Tick start = nn::os::GetSystemTick();

        // 送信
        size = nn::htcs::Send(socket, sendBuffer, sendByteCount, 0);
        EXPECT_EQ(sendByteCount, size);
        if (size < 0)
        {
            NN_LOG("L%d failed(%d)\n", __LINE__, nn::htcs::GetLastError());
        }

        // 受信
        size = nn::htcs::Recv(socket, recvBuffer, recvByteCount, nn::htcs::HTCS_MSG_WAITALL);
        EXPECT_EQ(recvByteCount, size);
        if (size < 0)
        {
            NN_LOG("L%d failed(%d)\n", __LINE__, nn::htcs::GetLastError());
        }

        nn::os::Tick end = nn::os::GetSystemTick();

        return nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
    }
}

// ノンブロッキングモードで受信データが無いときの受信レイテンシ測定
TEST_F(BenchmarkClient, NonBlocking_RecvNoData)
{
    // データの送受信回数
    const int repeatCount = 100;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    int n = nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);
    ASSERT_NE(-1, n);

    // --------------------------------------------------------------------------------
    uint8_t recvBuffer[1];

    // データが無い状態で受信
    double sum = 0.0;
    for (int i = 0; i < repeatCount; i++)
    {
        nn::os::Tick start = nn::os::GetSystemTick();

        auto size = nn::htcs::Recv(socket, recvBuffer, sizeof(recvBuffer), 0);
        EXPECT_EQ(-1, size);

        nn::os::Tick end = nn::os::GetSystemTick();

        sum += nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
    }

    double average = sum / repeatCount;
    NN_LOG("##teamcity[buildStatisticValue key='NonBlockingLatency_RecvNoData(us)' value='%f']\n", average * 1000.0 * 1000.0);
    // --------------------------------------------------------------------------------

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

// ノンブロッキングモードにおけるレイテンシの測定
TEST_F(BenchmarkClient, NonBlocking)
{
    // データの送受信回数
    const int repeatCount = 10;

    // バッファサイズ
    const int32_t bufferSize = repeatCount;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        NN_LOG("error:%d\n", nn::htcs::GetLastError());
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    int n = nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK);
    ASSERT_NE(-1, n);

    // --------------------------------------------------------------------------------
    uint8_t sendBuffer[bufferSize];
    uint8_t recvBuffer[bufferSize];
    nn::htcs::ssize_t size;

    // 送受信サイズを送信（ホストと同一エンディアンであると仮定）
    size = nn::htcs::Send(socket, &bufferSize, sizeof(int32_t), 0);
    EXPECT_EQ(sizeof(int32_t), size);
    size = nn::htcs::Send(socket, &bufferSize, sizeof(int32_t), 0);
    EXPECT_EQ(sizeof(int32_t), size);

    // Ack 受信
    uint8_t ack;
    size = nn::htcs::Recv(socket, &ack, 1, 0);

    // 1バイトずつ送信
    {
        double sum = 0.0;
        for (int i = 0; i < repeatCount; i++)
        {
            nn::os::Tick start = nn::os::GetSystemTick();

            size = nn::htcs::Send(socket, &sendBuffer[i], 1, 0);
            EXPECT_EQ(1, size);

            nn::os::Tick end = nn::os::GetSystemTick();

            double latency = nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
            NN_LOG("NonBlockingLatency_Send: %f us\n", latency * 1000.0 * 1000.0);

            sum += latency;
        }

        double average = sum / repeatCount;
        NN_LOG("##teamcity[buildStatisticValue key='NonBlockingLatency_Send(us)' value='%f']\n", average * 1000.0 * 1000.0);
    }

    // ノンブロッキングでも確実に受信できるよう、一定時間待つ
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(5) );

    // 1バイトずつ受信
    {
        double sum = 0.0;
        for (int i = 0; i < repeatCount; i++)
        {
            nn::os::Tick start = nn::os::GetSystemTick();

            size = nn::htcs::Recv(socket, &recvBuffer[i], 1, 0);
            EXPECT_EQ(1, size);

            nn::os::Tick end = nn::os::GetSystemTick();

            double latency = nn::os::ConvertToTimeSpan(end - start).GetNanoSeconds() / (1000.0 * 1000.0 * 1000.0);
            NN_LOG("NonBlockingLatency_Recv: %f us\n", latency * 1000.0 * 1000.0);

            sum += latency;
        }

        double average = sum / repeatCount;
        NN_LOG("##teamcity[buildStatisticValue key='NonBlockingLatency_Recv(us)' value='%f']\n", average * 1000.0 * 1000.0);
    }
    // --------------------------------------------------------------------------------

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

// 小さいサイズのデータを何度か送受信して、レイテンシを計算
TEST_F(BenchmarkClient, Latency)
{
    // データの送受信回数
    const int repeatCount = 100;

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    // --------------------------------------------------------------------------------
    uint8_t sendBuffer[1];
    uint8_t recvBuffer[1];

    double sum = 0.0;
    for (int i = 0; i < repeatCount; i++)
    {
        sum += ::BulkCommunication(socket, sendBuffer, sizeof(sendBuffer), recvBuffer, sizeof(recvBuffer));
    }
    double latency = sum / (repeatCount * 2); // ミリ秒

    NN_LOG("Recv count: %d\n", repeatCount);
    NN_LOG("Send count: %d\n", repeatCount);
    NN_LOG("Time: %f seconds\n", sum);
    NN_LOG("Latency: %f seconds\n", latency);

    NN_LOG("##teamcity[buildStatisticValue key='Latency(ms)' value='%f']\n", latency * 1000.0);
    EXPECT_GT(LatencyThreshold.at(g_Type), latency);
    // --------------------------------------------------------------------------------

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(BenchmarkClient, BandWidth)
{
    // 送受信サイズ
    const size_t minSize = 1024;
    const size_t maxSize = 64 * 1024 * 1024;
    const size_t ratio = 4; // サイズを増やす際の倍率

    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    // --------------------------------------------------------------------------------
    for( int size = minSize; size <= maxSize; size *= ratio )
    {
        uint8_t* recvBuffer = reinterpret_cast<uint8_t*>(malloc(size));
        uint8_t* sendBuffer = reinterpret_cast<uint8_t*>(malloc(size));
        ASSERT_NE(nullptr, sendBuffer);
        ASSERT_NE(nullptr, recvBuffer);

        for (int i = 0; i < size; i++)
        {
            recvBuffer[i] = 0;
            sendBuffer[i] = 0;
        }

        // 送信速度の測定
        {
            double duration = ::BulkCommunication(socket, sendBuffer, size, recvBuffer, 1);
            double bytePerSecond = (size + 1) / duration;

            NN_LOG("Send: %d bytes\n", size);
            NN_LOG("Time: %f sec\n", duration);
            NN_LOG("BandWidth: %f byte / sec\n", bytePerSecond);

            NN_LOG("##teamcity[buildStatisticValue key='BandWidth(MB/s)_TargetToHost_%dKB' value='%f']\n", size / 1024, bytePerSecond / (1000 * 1000));
            if( size >= 16 * 1024 * 1024)
            {
                EXPECT_LT(T2HBandWidthThreshold.at(g_Type), bytePerSecond);
            }
        }

        // 受信速度の測定
        {
            double duration = ::BulkCommunication(socket, sendBuffer, 1, recvBuffer, size);
            double bytePerSecond = (size + 1) / duration;

            NN_LOG("Recv: %d bytes\n", size);
            NN_LOG("Time: %f sec\n", duration);
            NN_LOG("BandWidth: %f byte / sec\n", bytePerSecond);

            NN_LOG("##teamcity[buildStatisticValue key='BandWidth(MB/s)_HostToTarget_%dKB' value='%f']\n", size / 1024, bytePerSecond / (1000 * 1000));
            if( size >= 16 * 1024 * 1024)
            {
                EXPECT_LT(H2TBandWidthThreshold.at(g_Type), bytePerSecond);
            }
        }

        free(recvBuffer);
        free(sendBuffer);
    }
    // --------------------------------------------------------------------------------

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(BenchmarkClient, End)
{
    // 接続したいホスト側サーバのアドレス情報構築
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "ServerInHost");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    NN_LOG("Trying to connect...\n");
    while (NN_STATIC_CONDITION(true))
    {
        if (nn::htcs::Connect(socket, &addr) == 0)
        {
            NN_LOG("Connected to server on host.\n");
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
    NN_LOG("Connected!\n");

    // 送受信サイズを両方 0 byte にすると、テストを終了する
    ::BulkCommunication(socket, nullptr, 0, nullptr, 0);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

extern "C" void nnMain()
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // ユーザー引数による設定
    ParseArguments(argc, argv);
    NN_LOG("ThresholdType : %s\n", StrThresholdType.at(g_Type));

    // GoogleTEST 初期化
    ::testing::InitGoogleTest(&argc, argv);
    // GoogleTEST 実行
    int testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);

    return;
}
