﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   Server のテスト
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/EchoClient を実行する必要があります。
*/

namespace
{
    // for MultipleAccept test
    const size_t                    SessionCount = nn::htcs::SessionCountMax;
    const size_t                    ThreadStackSize = 8192; // スレッド操作スレッドのスタックサイズ
    const size_t                    ThreadCount = SessionCount - 1;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadCount][ThreadStackSize];   // スレッドのスタック
    const char*                     PortName = "ServerInTarget";

    void AcceptThread(void *arg)
    {
        int* pSocket = static_cast<int*>( arg );
        int sock = nn::htcs::Accept(*pSocket, nullptr);
        EXPECT_EQ(-1, sock);
        EXPECT_EQ(nn::htcs::HTCS_EINTR, nn::htcs::GetLastError());
    }

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class BasicServer : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    // チェック用文字列を送信し、返ってきたデータが送信したものと同じであることを確認する
    void CheckEcho(int sock, const char *checkString)
    {
        nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, checkString, strlen(checkString), 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
            return;
        }
        // 受信側が ReadLine で読み込むため改行文字を送信
        nn::htcs::Send(sock, "\n", 1, 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
            return;
        }

        char buf[128];
        nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
        EXPECT_LT(0, receivedBytes);
        if (receivedBytes < 0)
        {
            NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
            return;
        }
        else if (receivedBytes == 0)
        {
            NN_LOG("gracefully closed\n");
            return;
        }

        buf[receivedBytes] = '\0';
        EXPECT_STREQ(checkString, buf);
    }

    void SendEnd(int sock)
    {
        const char closeMessage[] = "End\n";
        nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, closeMessage, strlen(closeMessage), 0);
        EXPECT_LE(0, sentBytes);
        if (sentBytes < 0)
        {
            NN_LOG("send error %d\n", nn::htcs::GetLastError());
        }
    }
}

TEST_F(BasicServer, Simple)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, sock);

    CheckEcho( sock, "Sending from Target to Host." );

    EXPECT_EQ(0, nn::htcs::Close(sock));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

#if 0
TEST_F(BasicServer, OutOfOrder)
{
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    const size_t CountMax = 31;
    int socketArray[CountMax];
    for (int i = 0; i < CountMax; i++)
    {
        socketArray[i] = nn::htcs::Socket();
        ASSERT_LE(0, socketArray[i]);
    }
    EXPECT_EQ(-1, nn::htcs::Accept(listenSocket, nullptr));
    EXPECT_EQ(nn::htcs::HTCS_EMFILE, nn::htcs::GetLastError());

    for (int i = 0; i < CountMax; i++)
    {
        EXPECT_EQ(0, nn::htcs::Close(socketArray[i]));
    }

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}
#endif

TEST_F(BasicServer, RepeatSendAndReceive)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, sock);

    for( int i = 0; i < 10; ++i )
    {
        char buffer[8];
        sprintf( buffer, "Echo %d", i );
        CheckEcho( sock, buffer );
    }

    EXPECT_EQ(0, nn::htcs::Close(sock));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(BasicServer, MultipleAccept)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, sock);

    nn::os::ThreadType thread[ThreadCount];
    for( int i = 0; i < ThreadCount; i++ )
    {
        nn::Result result = nn::os::CreateThread(&thread[i], AcceptThread, &listenSocket, g_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority);
        ASSERT_TRUE( result.IsSuccess() );
        nn::os::StartThread( &thread[i] );
    }

    // 各スレッドでの Accept を待つため数秒スリープ
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(5) );

    CheckEcho( sock, "Sending from Target to Host." );

    EXPECT_EQ(0, nn::htcs::Close(listenSocket));

    for( int i = 0; i < ThreadCount; i++ )
    {
        nn::os::WaitThread( &thread[i] );
        nn::os::DestroyThread( &thread[i] );
    }

    EXPECT_EQ(0, nn::htcs::Close(sock));
}

TEST_F(BasicServer, Peek)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, sock);

    const char* checkString = "Sending from Target to Host.";
    nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, checkString, strlen(checkString), 0);
    EXPECT_LE(0, sentBytes);
    if (sentBytes < 0)
    {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return;
    }
    // 受信側が ReadLine で読み込むため改行文字を送信
    nn::htcs::Send(sock, "\n", 1, 0);
    EXPECT_LE(0, sentBytes);
    if (sentBytes < 0)
    {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return;
    }

    char buf[128];
    for (int i = 0; i < 10; i++)
    {
        nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, nn::htcs::HTCS_MSG_PEEK);
        EXPECT_LT(0, receivedBytes);
        if (receivedBytes < 0)
        {
            NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
            return;
        }
        else if (receivedBytes == 0)
        {
            NN_LOG("gracefully closed\n");
            return;
        }

        buf[receivedBytes] = '\0';
        EXPECT_STREQ(checkString, buf);

        memset(buf, 0, sizeof(buf));
    }

    nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
    EXPECT_LT(0, receivedBytes);
    if (receivedBytes < 0)
    {
        NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
        return;
    }
    else if (receivedBytes == 0)
    {
        NN_LOG("gracefully closed\n");
        return;
    }

    buf[receivedBytes] = '\0';
    EXPECT_STREQ(checkString, buf);

    EXPECT_EQ(0, nn::htcs::Close(sock));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(BasicServer, End)
{
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, PortName);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, sock);

    SendEnd( sock );

    EXPECT_EQ(0, nn::htcs::Close(sock));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));}
