﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   単体テスト
*   事前に Target Manager を立ち上げておく必要があります。
*/

namespace
{
    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class ApiUnit : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };
}

TEST_F(ApiUnit, Socket)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    EXPECT_EQ(-1, nn::htcs::Close(-1));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Close(socket));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());

    const size_t CountMax = 32;
    int socketArray[CountMax];
    for (int i = 0; i < CountMax; i++)
    {
        socketArray[i] = nn::htcs::Socket();
        ASSERT_LE(0, socketArray[i]);
    }

    socket = nn::htcs::Socket();
    EXPECT_EQ(-1, socket);
    EXPECT_EQ(nn::htcs::HTCS_EMFILE, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socketArray[0]));
    socketArray[0] = nn::htcs::Socket();

    for (int i = 0; i < CountMax; i++)
    {
        EXPECT_EQ(0, nn::htcs::Close(socketArray[i]));
    }

    for (int i = 0; i < CountMax; i++)
    {
        socketArray[i] = nn::htcs::Socket();
        ASSERT_LE(0, socketArray[i]);
    }

    for (int i = 0; i < CountMax; i++)
    {
        EXPECT_EQ(0, nn::htcs::Close(socketArray[i]));
    }
}

TEST_F(ApiUnit, Connect)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "None");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    EXPECT_EQ(-1, nn::htcs::Connect(socket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_EADDRNOTAVAIL, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(-1, nn::htcs::Connect(socket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, BindAndListen)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "Server");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    EXPECT_EQ(-1, nn::htcs::Listen(socket, 1));
    EXPECT_EQ(nn::htcs::HTCS_EINVAL, nn::htcs::GetLastError());
    EXPECT_EQ(0, nn::htcs::Bind(socket, &addr));
    EXPECT_EQ(-1, nn::htcs::Bind(socket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_EINVAL, nn::htcs::GetLastError());
    EXPECT_EQ(0, nn::htcs::Listen(socket, 1));
    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Bind(socket, &addr));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
    EXPECT_EQ(-1, nn::htcs::Listen(socket, 1));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, Accept)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    EXPECT_EQ(-1, nn::htcs::Accept(socket, nullptr));
    EXPECT_EQ(nn::htcs::HTCS_EINVAL, nn::htcs::GetLastError());
    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(-1, nn::htcs::Accept(socket, nullptr));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, Recv)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    char buf[128];
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    EXPECT_EQ(-1, nn::htcs::Recv(socket, buf, sizeof(buf) - 1, 0));
    EXPECT_EQ(nn::htcs::HTCS_ENOTCONN, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Recv(socket, buf, sizeof(buf) - 1, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());

}

TEST_F(ApiUnit, Send)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    char buf[128];
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    EXPECT_EQ(-1, nn::htcs::Send(socket, buf, sizeof(buf), 0));
    EXPECT_EQ(nn::htcs::HTCS_ENOTCONN, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Send(socket, buf, sizeof(buf), 0));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, Shutdown)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    // [不具合？] 成功してしまう
    //EXPECT_EQ(-1, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RD));
    //EXPECT_EQ(nn::htcs::HTCS_ENOTCONN, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RD));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, Fcntl)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));

    EXPECT_NE(-1, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, 0));
    EXPECT_EQ(0, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));

    EXPECT_NE(-1, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));
    EXPECT_EQ(nn::htcs::HTCS_O_NONBLOCK, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));

    // TODO: fcntl の cmd の最大値に設定する（現状の 100 は適当な値）
    for (int i = 0; i < 100; i++)
    {
        if (i == nn::htcs::HTCS_F_GETFL || i == nn::htcs::HTCS_F_SETFL)
        {
            continue;
        }

        EXPECT_EQ(-1, nn::htcs::Fcntl(socket, i, 0));
        EXPECT_EQ(nn::htcs::HTCS_EINVAL, nn::htcs::GetLastError());
    }

    EXPECT_EQ(0, nn::htcs::Close(socket));

    EXPECT_EQ(-1, nn::htcs::Fcntl(socket, nn::htcs::HTCS_F_GETFL, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBADF, nn::htcs::GetLastError());
}

TEST_F(ApiUnit, GetPeerNameAny)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::HtcsPeerName name = nn::htcs::GetPeerNameAny();
    EXPECT_STREQ("", name.name);
}

TEST_F(ApiUnit, GetDefaultHostName)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    nn::htcs::HtcsPeerName name = nn::htcs::GetDefaultHostName();
    EXPECT_STREQ("", name.name);
}
