﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   メモリ使用量テスト
*/

namespace
{
    const size_t SocketCountMax = 32;
    size_t g_Size = 0;

    void* Allocate(size_t size)
    {
        g_Size += size;
        NN_LOG("%d / %d\n", g_Size, nn::htcs::GetWorkingMemorySize(SocketCountMax));
        EXPECT_GE(nn::htcs::GetWorkingMemorySize(SocketCountMax), g_Size);
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        g_Size -= size;
        free(p);
    }
}

TEST(Allocation, Basic)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    size_t previousSize;
    nn::htcs::Initialize(Allocate, Deallocate);
    previousSize = g_Size;

    int socket[SocketCountMax];
    for (int i = 0; i < SocketCountMax; i++)
    {
        NN_LOG("Loop(%d)\n", i);
        nn::htcs::SockAddrHtcs serviceAddr;
        serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
        serviceAddr.peerName = nn::htcs::GetPeerNameAny();
        nn::util::SNPrintf(serviceAddr.portName.name, sizeof(serviceAddr.portName.name), "AllocationTest%d", i);

        socket[i] = nn::htcs::Socket();
        EXPECT_LE(0, socket[i]);
        if (socket[i] < 0)
        {
            NN_LOG("Socket failed (%d)\n", nn::htcs::GetLastError());
        }
        int ret = nn::htcs::Bind(socket[i], &serviceAddr);
        EXPECT_EQ(0, ret);
        if (ret < 0)
        {
            NN_LOG("Bind failed (%d)\n", nn::htcs::GetLastError());
        }
        EXPECT_EQ(0, nn::htcs::Fcntl(socket[i], nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));
        EXPECT_EQ(-1, nn::htcs::Accept(socket[i], nullptr));
    }

    for (int i = 0; i < SocketCountMax; i++)
    {
        nn::htcs::Close(socket[i]);
    }

    nn::htcs::Finalize();
    EXPECT_EQ(0, g_Size);
}
