﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>

#include <nn/nn_Log.h>

namespace nnt { namespace htcs { namespace util {
    void Connect(int socket, nn::htcs::SockAddrHtcs* pSockAddrHtcs)
    {
        NN_LOG("Trying to connect %s ...\n", pSockAddrHtcs->portName.name);
        while (NN_STATIC_CONDITION(true))
        {
            if (nn::htcs::Connect(socket, pSockAddrHtcs) == 0)
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        }
        NN_LOG("Connected to %s.\n", pSockAddrHtcs->portName.name);
    }

    void ConnectToAnyHost(int socket, nn::htcs::HtcsPortName* pHtcsPortName)
    {
        // 接続したいホスト側サーバのアドレス情報構築
        nn::htcs::SockAddrHtcs addr;
        addr.family = nn::htcs::HTCS_AF_HTCS;
        addr.peerName = nn::htcs::GetPeerNameAny();
        std::memcpy(&addr.portName, pHtcsPortName, sizeof(addr.portName));

        util::Connect(socket, &addr);
    }

    void ConnectToTestServer(int socket)
    {
        nn::htcs::HtcsPortName portName;
        std::strcpy(portName.name, "HtcsTestManager");
        ConnectToAnyHost(socket, &portName);
    }

    void Echo(int socket, nn::htcs::ssize_t size, uint8_t* sendBuffer, uint8_t* recvBuffer, int recvFlag)
    {
        ASSERT_TRUE(Send(socket, size, sendBuffer));
        ASSERT_TRUE(Recv(socket, size, recvBuffer, recvFlag));

        // 検証
        int errorCount = 0;
        const int errorCountMax = 10;
        for (int i = 0; i < size; i++)
        {
            EXPECT_EQ(sendBuffer[i], recvBuffer[i]);
            if (sendBuffer[i] != recvBuffer[i])
            {
                errorCount++;
            }

            // size が大きいとエラー出力長くなりすぎるため、一定回数で打ち切る
            if (errorCount > errorCountMax)
            {
                NN_LOG("Error count was larger than %d. Verification is broken off.\n", errorCountMax);
                break;
            }
        }
    }

    bool Recv(int socket, nn::htcs::ssize_t size, uint8_t* recvBuffer, int flag)
    {
        nn::htcs::ssize_t recievedByteCount = 0;
        do
        {
            auto n = nn::htcs::Recv(socket, &recvBuffer[recievedByteCount], size - recievedByteCount, flag);
            if (n < 0 && nn::htcs::GetLastError() == nn::htcs::HTCS_EINTR)
            {
                // Retry because of EINTR
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                continue;
            }

            EXPECT_GT(n, 0);
            if (n < 0)
            {
                NN_LOG("LastError = %d\n", nn::htcs::GetLastError());
                return false;
            }
            recievedByteCount += n;
        } while (recievedByteCount < size);
        EXPECT_EQ(size, recievedByteCount);
        return true;
    }

    bool Send(int socket, nn::htcs::ssize_t size, uint8_t* sendBuffer)
    {
        nn::htcs::ssize_t sentByteCount = 0;
        do
        {
            auto n = nn::htcs::Send(socket, &sendBuffer[sentByteCount], size - sentByteCount, 0);
            if (n < 0 && nn::htcs::GetLastError() == nn::htcs::HTCS_EINTR)
            {
                // Retry because of EINTR
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                continue;
            }

            EXPECT_GT(n, 0);
            if (n < 0)
            {
                NN_LOG("LastError = %d\n", nn::htcs::GetLastError());
                return false;
            }
            sentByteCount += n;
        } while (sentByteCount < size);
        EXPECT_EQ(size, sentByteCount);
        return true;
    }
}}}
