﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nnt/nntest.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_DebugApi.h>

#include "../../../../../Programs/Eris/Sources/Libraries/htclow/server/htclow_Packet.h"

#include "../testHtclow_Util.h"
#include "testHtclow_PacketUtil.h"

namespace nnt { namespace htclow {

void AssertSynPacket(const nn::htclow::server::ReceivePacket& packet, nn::htclow::detail::ChannelInternalType channel) NN_NOEXCEPT
{
    const auto header = packet.GetHeader();
    ASSERT_EQ(header->protocol, nn::htclow::server::ProtocolId);
    ASSERT_EQ(header->version, nn::htclow::server::MaxVersion);
    ASSERT_EQ(header->packetType, nn::htclow::server::PacketType::Syn);
    ASSERT_EQ(header->channel.moduleId, channel.moduleId);
    ASSERT_EQ(header->channel.channelId, channel.channelId);
    ASSERT_EQ(header->bodySize, 0);
    ASSERT_EQ(header->sequenceId, 0L);
}

void AssertDataPacket(const nn::htclow::server::ReceivePacket& packet, nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId, const void* body, int bodySize) NN_NOEXCEPT
{
    const auto header = packet.GetHeader();
    ASSERT_EQ(nn::htclow::server::ProtocolId, header->protocol);
    ASSERT_EQ(nn::htclow::server::MaxVersion, header->version);
    ASSERT_EQ(nn::htclow::server::PacketType::Data, header->packetType);
    ASSERT_EQ(channel.moduleId, header->channel.moduleId);
    ASSERT_EQ(channel.channelId, header->channel.channelId);
    ASSERT_EQ(bodySize, header->bodySize);
    ASSERT_EQ(sequenceId, header->sequenceId);

    ASSERT_EQ(0, std::memcmp(packet.GetBody(), body, bodySize));
}

void AssertDataAckPacket(const nn::htclow::server::ReceivePacket& packet, nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    const auto header = packet.GetHeader();
    ASSERT_EQ(nn::htclow::server::ProtocolId, header->protocol);
    ASSERT_EQ(nn::htclow::server::MaxVersion, header->version);
    ASSERT_EQ(nn::htclow::server::PacketType::DataAck, header->packetType);
    ASSERT_EQ(channel.moduleId, header->channel.moduleId);
    ASSERT_EQ(channel.channelId, header->channel.channelId);
    ASSERT_EQ(0, header->bodySize);
    ASSERT_EQ(sequenceId, header->sequenceId);
}

void AssertFinPacket(const nn::htclow::server::ReceivePacket& packet, nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    const auto header = packet.GetHeader();
    ASSERT_EQ(nn::htclow::server::ProtocolId, header->protocol);
    ASSERT_EQ(nn::htclow::server::MaxVersion, header->version);
    ASSERT_EQ(nn::htclow::server::PacketType::Fin, header->packetType);
    ASSERT_EQ(channel.moduleId, header->channel.moduleId);
    ASSERT_EQ(channel.channelId, header->channel.channelId);
    ASSERT_EQ(0, header->bodySize);
    ASSERT_EQ(sequenceId, header->sequenceId);
}

void AssertFinAckPacket(const nn::htclow::server::ReceivePacket& packet, nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    const auto header = packet.GetHeader();
    ASSERT_EQ(nn::htclow::server::ProtocolId, header->protocol);
    ASSERT_EQ(nn::htclow::server::MaxVersion, header->version);
    ASSERT_EQ(nn::htclow::server::PacketType::FinAck, header->packetType);
    ASSERT_EQ(channel.moduleId, header->channel.moduleId);
    ASSERT_EQ(channel.channelId, header->channel.channelId);
    ASSERT_EQ(0, header->bodySize);
    ASSERT_EQ(sequenceId, header->sequenceId);
}

std::unique_ptr<nn::htclow::server::SendPacket> MakeDataPacket(nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId, const void* body, int bodySize) NN_NOEXCEPT
{
    auto packet = std::make_unique<nn::htclow::server::SendPacket>(static_cast<int>(sizeof(nn::htclow::server::PacketHeader)) + bodySize);

    packet->GetHeader()->protocol = nn::htclow::server::ProtocolId;
    packet->GetHeader()->version = nn::htclow::server::MaxVersion;
    packet->GetHeader()->reserved = 0;
    packet->GetHeader()->packetType = nn::htclow::server::PacketType::Data;
    packet->GetHeader()->channel = channel;
    packet->GetHeader()->bodySize = bodySize;
    packet->GetHeader()->sequenceId = sequenceId;

    if (bodySize > 0)
    {
        packet->CopyBody(body, bodySize);
    }

    return packet;
}

std::unique_ptr<nn::htclow::server::SendPacket> MakeDataAckPacket(nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    auto packet = std::make_unique<nn::htclow::server::SendPacket>();

    packet->GetHeader()->protocol = nn::htclow::server::ProtocolId;
    packet->GetHeader()->version = nn::htclow::server::MaxVersion;
    packet->GetHeader()->reserved = 0;
    packet->GetHeader()->packetType = nn::htclow::server::PacketType::DataAck;
    packet->GetHeader()->channel = channel;
    packet->GetHeader()->bodySize = 0;
    packet->GetHeader()->sequenceId = sequenceId;

    return packet;
}

std::unique_ptr<nn::htclow::server::SendPacket> MakeSynAckPacket(nn::htclow::detail::ChannelInternalType channel) NN_NOEXCEPT
{
    auto packet = std::make_unique<nn::htclow::server::SendPacket>();

    packet->GetHeader()->protocol = nn::htclow::server::ProtocolId;
    packet->GetHeader()->version = nn::htclow::server::MaxVersion;
    packet->GetHeader()->reserved = 0;
    packet->GetHeader()->packetType = nn::htclow::server::PacketType::SynAck;
    packet->GetHeader()->channel = channel;
    packet->GetHeader()->bodySize = 0;
    packet->GetHeader()->sequenceId = 0L;

    return packet;
}

std::unique_ptr<nn::htclow::server::SendPacket> MakeFinPacket(nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    auto packet = std::make_unique<nn::htclow::server::SendPacket>();

    packet->GetHeader()->protocol = nn::htclow::server::ProtocolId;
    packet->GetHeader()->version = nn::htclow::server::MaxVersion;
    packet->GetHeader()->reserved = 0;
    packet->GetHeader()->packetType = nn::htclow::server::PacketType::Fin;
    packet->GetHeader()->channel = channel;
    packet->GetHeader()->bodySize = 0;
    packet->GetHeader()->sequenceId = sequenceId;

    return packet;
}

std::unique_ptr<nn::htclow::server::SendPacket> MakeFinAckPacket(nn::htclow::detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
{
    auto packet = std::make_unique<nn::htclow::server::SendPacket>();

    packet->GetHeader()->protocol = nn::htclow::server::ProtocolId;
    packet->GetHeader()->version = nn::htclow::server::MaxVersion;
    packet->GetHeader()->reserved = 0;
    packet->GetHeader()->packetType = nn::htclow::server::PacketType::FinAck;
    packet->GetHeader()->channel = channel;
    packet->GetHeader()->bodySize = 0;
    packet->GetHeader()->sequenceId = sequenceId;

    return packet;
}

std::unique_ptr<nn::htclow::server::ReceivePacket> ReceivePacketForTest() NN_NOEXCEPT
{
    nn::htclow::server::PacketHeader header;
    nn::htclow::detail::ReceiveFromTest(&header, sizeof(header));

    auto packet = std::make_unique<nn::htclow::server::ReceivePacket>(static_cast<int>(sizeof(header)) + header.bodySize);
    packet->CopyHeader(&header);

    if (header.bodySize > 0)
    {
        nn::htclow::detail::ReceiveFromTest(packet->GetBody(), header.bodySize);
    }

    return packet;
}

void SendPacketForTest(const nn::htclow::server::SendPacket& packet) NN_NOEXCEPT
{
    nn::htclow::detail::SendFromTest(packet.GetBuffer(), packet.GetBufferSize());
}

}}
