﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nn/hidbus/hidbus.h>

namespace
{
    // テスト FW の実行コマンド定義
    enum TestFwDataName
    {
        TestFwDataName_LoopBack       = 0x00,
        TestFwDataName_InvalidCrcData = 0x01,
        TestFwDataName_Timeout        = 0x02,
    };

    // Sherry Test FW の ID
    const int TestFwId = 0xff;

    // SubThread のスタック
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_SubThreadStack[nn::os::MemoryPageSize];

    // NpadID のリスト
    const nn::hid::NpadIdType NpadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
    };

    // Npad 周りのセットアップ関数
    void SetUpNpad()
    {
        // Npad の初期化
        nn::hid::InitializeNpad();

        // Style の設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyRight::Mask);

        // サポートする Npad ID のリスト
        nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    }

    void SendAndReceiveSubThread(void* arg)
    {
        nn::hidbus::BusHandle subThreadHandle;

        nn::hid::NpadIdType subThreadNpadId = *reinterpret_cast<nn::hid::NpadIdType*>(arg);

        // ハンドルの取得
        while (!nn::hidbus::GetBusHandle(&subThreadHandle, subThreadNpadId, nn::hidbus::BusType_RightJoyRail))
        {
            NN_LOG("[hidbus] Please Connect Right Joy-Con(2)\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NN_LOG("[hidbus] handle = %llu(2)\n", subThreadHandle);

        // 初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(subThreadHandle));

        // Enable
        while (nn::hidbus::EnableExternalDevice(true, TestFwId, subThreadHandle).IsFailure())
        {
            NN_LOG("[hidbus] Please Connect Sherry Device(2)\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        for (int i = 0; i < 28; i++)
        {
            size_t outDataSize = 0;
            uint8_t outBuffer[32];
            uint8_t inBuffer[32] = { 0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
                1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
                11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
                21, 22, 23, 24, 25, 26, 27, 28 };

            // 4 byte 目に送るデータのサイズを入れる。
            inBuffer[3] = static_cast<uint8_t>(i + 1);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), subThreadHandle, inBuffer, 4 + 1 + i));
            NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, outDataSize);
            for (int l = 0; l < 4; l++)
            {
                ASSERT_EQ(0x00, outBuffer[l]);
                NN_LOG("%d ", outBuffer[l]);
            }
            NN_LOG("(2)\n");
            for (int m = 4; m < 4 + 1 + i; m++)
            {
                ASSERT_EQ(inBuffer[m], outBuffer[m]);
            }
        }

        // Disable
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, subThreadHandle));

        // Finalize
        nn::hidbus::Finalize(subThreadHandle);
    }
}

/**
* TestFw に対する MultiThread アクセステスト：SendAndReceive 周り(右ジョイコン及び TestFw が書かれた Sherry デバイスが必要)
*/
TEST(HidbusMultiThreadTest, SendAndReceive)
{
    SetUpNpad();

    // mainThread の handle
    nn::hidbus::BusHandle mainThreadHandle = {};
    nn::os::ThreadType    subThread;

    // Sub スレッドでアクセスする NpadID
    nn::hid::NpadIdType subThreadNpadId = nn::hid::NpadId::No2;

    // スレッドの作成を試行
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::os::CreateThread(&subThread,
        SendAndReceiveSubThread,
        &subThreadNpadId,
        s_SubThreadStack,
        sizeof(s_SubThreadStack),
        nn::os::DefaultThreadPriority));

    // スレッドの開始
    nn::os::StartThread(&subThread);

    NN_UTIL_SCOPE_EXIT
    {
        // スレッドの終了を待つ
        nn::os::WaitThread(&subThread);
        nn::os::DestroyThread(&subThread);
    };

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&mainThreadHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", mainThreadHandle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(mainThreadHandle));

    // Enable
    while (NN_STATIC_CONDITION(true))
    {
        auto enableResult = nn::hidbus::EnableExternalDevice(true, TestFwId, mainThreadHandle);
        if (enableResult.IsSuccess())
        {
            break;
        }
        else
        {
            NN_LOG("[hidbus] Please Connect Sherry Device\n");
            NN_LOG("[hidbus] Result = %d, %d\n", enableResult.GetModule(), enableResult.GetInnerValueForDebug());

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

    }
    while (nn::hidbus::EnableExternalDevice(true, TestFwId, mainThreadHandle).IsFailure())
    {
        NN_LOG("[hidbus] Please Connect Sherry Device\n");
        NN_LOG("[hidbus] Result \n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    for (int i = 0; i < 28; i++)
    {
        size_t outDataSize = 0;
        uint8_t outBuffer[32];
        uint8_t inBuffer[32] = {0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
                                 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
                                11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
                                21, 22, 23, 24, 25, 26, 27, 28};

        // 4 byte 目に送るデータのサイズを入れる。
        inBuffer[3] = static_cast<uint8_t>(i + 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), mainThreadHandle, inBuffer, 4 + 1 + i ));
        NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, outDataSize);
        for (int l = 0; l < 4; l++)
        {
            ASSERT_EQ( 0x00, outBuffer[l]);
            NN_LOG("%d ", outBuffer[l]);
        }
        NN_LOG("\n");
        for (int m = 4; m < 4 + 1 + i; m++)
        {
            ASSERT_EQ(inBuffer[m], outBuffer[m]);
        }
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, mainThreadHandle));

    // Finalize
    nn::hidbus::Finalize(mainThreadHandle);
}

/**
* TestFw に対する MultiThread アクセステスト：SendAndReceive 周り(右ジョイコン及び TestFw が書かれた Sherry デバイスが必要。同じ Npad ID でアクセスする)
*/
TEST(HidbusMultiThreadTest, SendAndReceiveWithSameNpadId)
{
    SetUpNpad();

    // mainThread の handle
    nn::hidbus::BusHandle mainThreadHandle = {};
    nn::os::ThreadType    subThread;

    // Sub スレッドでアクセスする NpadID
    nn::hid::NpadIdType subThreadNpadId = nn::hid::NpadId::No1;

    // スレッドの作成を試行
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::os::CreateThread(&subThread,
        SendAndReceiveSubThread,
        &subThreadNpadId,
        s_SubThreadStack,
        sizeof(s_SubThreadStack),
        nn::os::DefaultThreadPriority));

    // スレッドの開始
    nn::os::StartThread(&subThread);

    NN_UTIL_SCOPE_EXIT
    {
        // スレッドの終了を待つ
        nn::os::WaitThread(&subThread);
        nn::os::DestroyThread(&subThread);
    };

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&mainThreadHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", mainThreadHandle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(mainThreadHandle));

    // Enable
    while (NN_STATIC_CONDITION(true))
    {
        auto enableResult = nn::hidbus::EnableExternalDevice(true, TestFwId, mainThreadHandle);
        if (enableResult.IsSuccess())
        {
            break;
        }
        else
        {
            NN_LOG("[hidbus] Please Connect Sherry Device\n");
            NN_LOG("[hidbus] Result = %d, %d\n", enableResult.GetModule(), enableResult.GetInnerValueForDebug());

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

    }
    while (nn::hidbus::EnableExternalDevice(true, TestFwId, mainThreadHandle).IsFailure())
    {
        NN_LOG("[hidbus] Please Connect Sherry Device\n");
        NN_LOG("[hidbus] Result \n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    for (int i = 0; i < 28; i++)
    {
        size_t outDataSize = 0;
        uint8_t outBuffer[32];
        uint8_t inBuffer[32] = { 0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
            1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
            11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
            21, 22, 23, 24, 25, 26, 27, 28 };

        // 4 byte 目に送るデータのサイズを入れる。
        inBuffer[3] = static_cast<uint8_t>(i + 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), mainThreadHandle, inBuffer, 4 + 1 + i));
        NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, outDataSize);
        for (int l = 0; l < 4; l++)
        {
            ASSERT_EQ(0x00, outBuffer[l]);
            NN_LOG("%d ", outBuffer[l]);
        }
        NN_LOG("\n");
        for (int m = 4; m < 4 + 1 + i; m++)
        {
            ASSERT_EQ(inBuffer[m], outBuffer[m]);
        }
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, mainThreadHandle));

    // Finalize
    nn::hidbus::Finalize(mainThreadHandle);
}
