﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nn/hidbus/hidbus.h>

namespace
{
    // テスト FW の実行コマンド定義
    enum TestFwDataName
    {
        TestFwDataName_LoopBack       = 0x00,
        TestFwDataName_InvalidCrcData = 0x01,
        TestFwDataName_Timeout        = 0x02,
    };

    // PollingMode 時に渡すバッファ
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_Buffer[nn::os::MemoryPageSize];

    // Sherry Test FW の ID
    const int TestFwId = 0xff;

    // hidbus の handle (テストで共通で使用)
    nn::hidbus::BusHandle g_Handle = {};


    // NpadID のリスト
    const nn::hid::NpadIdType NpadIds[] =
    {
        nn::hid::NpadId::No1,
    };

    // Npad 周りのセットアップ関数
    void SetUpNpad()
    {
        // Npad の初期化
        nn::hid::InitializeNpad();

        // Style の設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyRight::Mask);

        // サポートする Npad ID のリスト
        nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    }
}

/**
* RONDE に対する基本的なテスト：Initialize 周り(右ジョイコンが必要)
*/
TEST(RondeBasicTest, Initialize)
{
    int count = 0;

    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化と終了を繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat Initialize and Finalize\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));
        nn::hidbus::Finalize(g_Handle);
    }

    // 初期化を複数回行う
    count = 0;
    NN_LOG("[hidbus] Repeat Initialize only\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));
    }

    // 終了を複数回行う
    count = 0;
    NN_LOG("[hidbus] Repeat Finalize only\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        nn::hidbus::Finalize(g_Handle);
    }
}

/**
* RONDE に対する基本的なテスト：Enable 周り(右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, Enable)
{
    int count = 0;

    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable と Disable を繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat Enable and Disable\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));
    }

    // Enable だけを繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat Enable\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));
    }

    // Disable だけを繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat Disable\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));
    }

    // 終了
    nn::hidbus::Finalize(g_Handle);
}

/**
* RONDE に対する基本的なテスト：PollingMode 周り(右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, EnableDisablePollingMode)
{
    int count = 0;

    const uint8_t RondePollingDataCommand[4] = { 0x01, 0x01, 0x02, 0x00 };

    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // EnablePollingMode と DisablePollingMode を繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat EnablePollingMode and DisablePollingMode\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, RondePollingDataCommand, sizeof(RondePollingDataCommand),
                                                                               s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));
    }

    // EnablePollingMode のみを繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat EnablePollingMode Only\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, RondePollingDataCommand, sizeof(RondePollingDataCommand),
            s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));
    }

    // DisablePollingMode のみを繰り返す
    count = 0;
    NN_LOG("[hidbus] Repeat DisablePollingMode Only\n");
    for (int i = 0; i < 10; i++)
    {
        count++;
        NN_LOG("[hidbus] Repeat Count = %d\n", count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // 終了
    nn::hidbus::Finalize(g_Handle);
}


/**
* RONDE に対する基本的なテスト：終了処理周り(右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, Finalize)
{
    const uint8_t RondePollingDataCommand[4] = { 0x01, 0x01, 0x02, 0x00 };

    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    NN_LOG("[hidbus] Start Finalize test (Skip DisablePollingMode and DisableDevice)\n");

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // EnablePollingMode
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, RondePollingDataCommand, sizeof(RondePollingDataCommand),
        s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));

    // Disable と DisablePollingMode を飛ばして、 Finalize する
    nn::hidbus::Finalize(g_Handle);

    NN_LOG("[hidbus] Done Finalize test (Skip DisablePollingMode and DisableDevice)\n");

    NN_LOG("[hidbus] Start Finalize test (Skip DisablePollingMode)\n");

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // EnablePollingMode
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, RondePollingDataCommand, sizeof(RondePollingDataCommand),
        s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));

    // DisablePollingMode を飛ばして、 Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);

    NN_LOG("[hidbus] Done Finalize test (Skip DisablePollingMode)\n");

    NN_LOG("[hidbus] Start Finalize test (Skip DisableDevice)\n");

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // Disable せずに Finalize
    nn::hidbus::Finalize(g_Handle);

    NN_LOG("[hidbus] Done Finalize test (Skip DisableDevice)\n");
}

/**
* RONDE に対する基本的なテスト：SendAndReceive 周り(右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, SendAndReceive)
{
    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    for (int i = 0; i < 28; i++)
    {
        size_t outDataSize = 0;
        uint8_t outBuffer[32];
        uint8_t inBuffer[32] = {0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
                                 1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
                                11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
                                21, 22, 23, 24, 25, 26, 27, 28};

        // 4 byte 目に送るデータのサイズを入れる。
        inBuffer[3] = static_cast<uint8_t>(i + 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), g_Handle, inBuffer, 4 + 1 + i ));
        NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, outDataSize);
        for (int l = 0; l < 4; l++)
        {
            ASSERT_EQ( 0x00, outBuffer[l]);
            NN_LOG("%d ", outBuffer[l]);
        }
        NN_LOG("\n");
        for (int m = 4; m < 4 + 1 + i; m++)
        {
            ASSERT_EQ(inBuffer[m], outBuffer[m]);
        }

        // デバッグ用
        /*
        NN_LOG(" inData = ");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
            NN_LOG("%2d ", inBuffer[j]);
        }
        NN_LOG("\n");

        NN_LOG("outData = ");
        for (int k = 4; k < 4 + 1 + i; k++)
        {
            NN_LOG("%2d ", outBuffer[k]);
        }
        NN_LOG("\n");
        */
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}

/**
* RONDE に対する基本的なテスト：SendAndReceive での InvalidCRC (右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, InvalidCrc)
{
    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    size_t outDataSize = 0;
    uint8_t outBuffer[32];
    uint8_t inBuffer[4] = { 0xf0, TestFwDataName_InvalidCrcData, 0x01, 0x00 };  // Test 用の SendComandHeader

    // 古いジョイ FW なら、 Timeout、新しいジョイ FW なら Success が返りつつ、データサイズが 0 になるはず
    auto result = nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), g_Handle, inBuffer, sizeof(inBuffer));
    if (result.IsSuccess())
    {
        ASSERT_EQ(0, outDataSize);
    }
    else
    {
        ASSERT_TRUE(nn::hidbus::ResultExternalDeviceTimeout::Includes(result));
    }


    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}

/**
* RONDE に対する基本的なテスト：SendAndReceive での Timeout (右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, Timeout)
{
    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    size_t outDataSize = 0;
    uint8_t outBuffer[32];
    uint8_t inBuffer[4] = { 0xf0, TestFwDataName_Timeout, 0x01, 0x00 };  // Test 用の SendComandHeader

    auto result = nn::hidbus::SendAndReceive(&outDataSize, &outBuffer, sizeof(outBuffer), g_Handle, inBuffer, sizeof(inBuffer));
    ASSERT_TRUE(nn::hidbus::ResultExternalDeviceTimeout::Includes(result));

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}

/**
* RONDE に対する基本的なテスト：PollingModeSixAxisSensorEnable 周り(右ジョイコン及び RONDE がささった状態で実行すること)
*/
TEST(RondeBasicTest, PollingModeSixAxisSensorEnable)
{
    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // SixAxisSensorEnable モードのテスト
    // MAX 8 byte なので、ヘッダーを除いて 4 byte まで受信できるはず
    for (int i = 0; i < 4; i++)
    {
        nn::hidbus::JoyPollingReceivedData receiveData[10];
        uint8_t inBuffer[32] = { 0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
            1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
            11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
            21, 22, 23, 24, 25, 26, 27, 28 };

        // 4 byte 目に送るデータのサイズを入れる。
        inBuffer[3] = static_cast<uint8_t>(i + 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, inBuffer, 4 + i + 1,
            s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));

        // 適当に 100ms 後の値を取得する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        // 最新１つをまずは取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(&receiveData[0], g_Handle));

        NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, receiveData[0].outSize);
        for (int j = 0; j < 4; j++)
        {
            ASSERT_EQ(0x00, receiveData[0].data[j]);
            NN_LOG("%d ", receiveData[0].data[j]);
        }
        NN_LOG("\n");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
            ASSERT_EQ(inBuffer[j], receiveData[0].data[j]);
        }

        // デバッグ用
        NN_LOG(" inData = ");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
        NN_LOG("%2d ", inBuffer[j]);
        }
        NN_LOG("\n");

        NN_LOG("outData = ");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
        NN_LOG("%2d ", receiveData[0].data[j]);
        }
        NN_LOG("\n");

        // 複数取得のテスト
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(receiveData, 10, g_Handle));
        for (int k = 0; k < 10; k++)
        {
            NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, receiveData[k].outSize);
            for (int j = 0; j < 4; j++)
            {
                ASSERT_EQ(0x00, receiveData[k].data[j]);
                NN_LOG("%d ", receiveData[k].data[j]);
            }
            NN_LOG("\n");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                ASSERT_EQ(inBuffer[j], receiveData[k].data[j]);
            }

            // デバッグ用
            NN_LOG(" inData = ");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                NN_LOG("%2d ", inBuffer[j]);
            }
            NN_LOG("\n");

            NN_LOG("outData = ");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                NN_LOG("%2d ", receiveData[k].data[j]);
            }
            NN_LOG("\n");
        }


        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}

TEST(RondeBasicTest, PollingModeSixAxisSensorDisable)
{
    SetUpNpad();

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // SixAxisSensorDisable モードのテスト
    // MAX 38 byte なので、ヘッダーを除いて 34 byte まで受信できるはずだが、
    // SendCommand の限界が 37 byte なので、ここでは 33 byte まで実行する (境界値は FW 側に別の機能を追加する)
    for (int i = 0; i < 33; i++)
    {
        nn::hidbus::JoyPollingReceivedData receiveData[10];
        uint8_t inBuffer[38] = { 0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
            1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
            11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
            21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
            31, 32, 33, 34};

        // 4 byte 目に送るデータのサイズを入れる。
        inBuffer[3] = static_cast<uint8_t>(i + 1);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, inBuffer, 4 + i + 1,
            s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorDisable));

        // 適当に 100ms 後の値を取得する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        // 最新１つをまずは取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(&receiveData[0], g_Handle));

        NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, receiveData[0].outSize);
        for (int j = 0; j < 4; j++)
        {
            ASSERT_EQ(0x00, receiveData[0].data[j]);
            NN_LOG("%d ", receiveData[0].data[j]);
        }
        NN_LOG("\n");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
            ASSERT_EQ(inBuffer[j], receiveData[0].data[j]);
        }

        // デバッグ用
        NN_LOG(" inData = ");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
            NN_LOG("%2d ", inBuffer[j]);
        }
        NN_LOG("\n");

        NN_LOG("outData = ");
        for (int j = 4; j < 4 + 1 + i; j++)
        {
            NN_LOG("%2d ", receiveData[0].data[j]);
        }
        NN_LOG("\n");

        // 複数取得のテスト
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(receiveData, 10, g_Handle));
        for (int k = 0; k < 10; k++)
        {
            NN_LOG("[hidbus] [%2d] outDataSize =%d, OutResult = ", i, receiveData[k].outSize);
            for (int j = 0; j < 4; j++)
            {
                ASSERT_EQ(0x00, receiveData[k].data[j]);
                NN_LOG("%d ", receiveData[k].data[j]);
            }
            NN_LOG("\n");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                ASSERT_EQ(inBuffer[j], receiveData[k].data[j]);
            }

            // デバッグ用
            NN_LOG(" inData = ");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                NN_LOG("%2d ", inBuffer[j]);
            }
            NN_LOG("\n");

            NN_LOG("outData = ");
            for (int j = 4; j < 4 + 1 + i; j++)
            {
                NN_LOG("%2d ", receiveData[k].data[j]);
            }
            NN_LOG("\n");
        }


        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));
    }

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}

TEST(RondeBasicTest, WithoutPollingModeEnable)
{
    SetUpNpad();

    nn::hidbus::JoyPollingReceivedData receiveData;
    uint8_t inBuffer[38] = { 0xf0, TestFwDataName_LoopBack, 0x01, 0x00,  // Test 用の SendComandHeader
        1,  2,  3,  4,  5,  6,  7,  8,  9, 10,
        11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
        21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
        31, 32, 33, 34 };

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&g_Handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", g_Handle);

    // 初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::Initialize(g_Handle));

    // Enable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(true, TestFwId, g_Handle));

    // PollingMode を Enable にせずに GetJoyPollingReceivedData を呼び出して dataSize が 0 になっていることの確認
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(&receiveData, g_Handle));
    ASSERT_EQ(0, receiveData.outSize);

    // 一度、 Enable にした後、 Disable にして、同じように dataSize が 0 になっていることの確認
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableJoyPollingReceiveMode(g_Handle, inBuffer, 8,
        s_Buffer, sizeof(s_Buffer), nn::hidbus::JoyPollingMode_SixAxisSensorEnable));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::DisableJoyPollingReceiveMode(g_Handle));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::GetJoyPollingReceivedData(&receiveData, g_Handle));
    ASSERT_EQ(0, receiveData.outSize);

    // Disable
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, TestFwId, g_Handle));

    // Finalize
    nn::hidbus::Finalize(g_Handle);
}
