﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "ShakeMotionDetector.h"
#include "SixAxisSensorGestureProcessor.h"

namespace {

} // namespace

ShakeMotionDetector::ShakeMotionDetector() NN_NOEXCEPT
    : m_RawState()
    , m_ShakeCount(0)
    , m_ThresholdShakeCount(5)
    , m_ThresholdShakeResetSampleCount(100) // 100 * interval でリセット
    , m_ThresholdAccelerometerLengthSquared(2.f)
    , m_ThresholdCos(0.f)
{
    // 何もしない
}

ShakeMotionDetector::~ShakeMotionDetector() NN_NOEXCEPT
{
    // 何もしない
}

void ShakeMotionDetector::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
}

void ShakeMotionDetector::Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT
{
    if (rawState.samplingNumber - m_RawState.samplingNumber > m_ThresholdShakeResetSampleCount)
    {
        m_RawState = rawState;
        m_ShakeCount = 0;
        return;
    }

    ::nn::util::Vector3f currentAccNormalized;
    auto currentAccLengthSquared = ::nn::util::VectorNormalize(&currentAccNormalized, rawState.acc);
    if (currentAccLengthSquared <= m_ThresholdAccelerometerLengthSquared)
    {
        return;
    }

    ::nn::util::Vector3f previousAccNormalized;
    auto previousAccLengthSquared = ::nn::util::VectorNormalize(&previousAccNormalized, m_RawState.acc);
    if (previousAccLengthSquared <= 0.f)
    {
        return;
    }

    auto cos = ::nn::util::VectorDot(currentAccNormalized, previousAccNormalized);

    if (cos > m_ThresholdCos)
    {
        return;
    }

    // NN_LOG("ShakeMotionDetected : %lld, %lld, %d, %f, %f, %f\n"
    //     , rawState.samplingNumber
    //     , m_SamplingNumber
    //     , m_ShakeCount
    //     , currentAccLengthSquared
    //     , previousAccLengthSquared
    //     , cos
    // );

    if (m_ThresholdShakeCount < m_ShakeCount)
    {
        m_Lifo.Append(
            MotionState({
                m_SamplingNumber,
                rawState.samplingNumber,
                MotionType::MotionType_Shake
                })
        );

        m_SamplingNumber++;
    }

    m_ShakeCount++;

    m_RawState = rawState;
}

int ShakeMotionDetector::GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT
{
    return m_Lifo.Read(pOutStates, count);
}
