﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "RotateMotionDetector.h"
#include "SixAxisSensorGestureProcessor.h"

namespace {

const float AngularVelocityConvertUnitCoef = 360.f;

} // namespace

RotateMotionDetector::RotateMotionDetector() NN_NOEXCEPT
    : m_RotationState()
    , m_RotationStateSamplingNumber(0)
    , m_SamplingIntervalSeconds(1.f / 60.f) // 60fps
    , m_ThresholdRotationStartedLengthSquared(5.f)
    , m_IsRotationStarted(false)
    , m_RotationStartedState()
{
    m_RotationStateLifo.Clear();
}

RotateMotionDetector::~RotateMotionDetector() NN_NOEXCEPT
{
    // 何もしない
}

void RotateMotionDetector::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
    m_RotationStateLifo.Clear();
}

void RotateMotionDetector::Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT
{
    auto gyroLengthSquared = rawState.gyro.LengthSquared();

    // NN_LOG("%f, %f, %f, %f\n"
    //     , gyroLengthSquared
    //     , rawState.gyro.GetX()
    //     , rawState.gyro.GetY()
    //     , rawState.gyro.GetZ()
    // );

    if (!m_IsRotationStarted &&
        gyroLengthSquared > m_ThresholdRotationStartedLengthSquared)
    {
        // NN_LOG("%s: Rotation started\n", __FUNCTION__);
        m_RotationStartedState = m_RotationState;
        m_IsRotationStarted = true;
    }
    else if(m_IsRotationStarted &&
            gyroLengthSquared < m_ThresholdRotationStartedLengthSquared)
    {
        // 回転終了

        m_IsRotationStarted = false;
        ::nn::util::Vector3f rotationDelta;
        ::nn::util::VectorSubtract(&rotationDelta,
                                   m_RotationState.rotation,
                                   m_RotationStartedState.rotation);

        // X
        auto maxAbsValue = std::abs(rotationDelta.GetX());
        auto maxAxis = 0;
        bool isPositive = (0 < rotationDelta.GetX());

        // Y
        auto tmp = std::abs(rotationDelta.GetY());
        if (maxAbsValue < tmp)
        {
            maxAbsValue = tmp;
            maxAxis = 1;
            isPositive = (0.f < rotationDelta.GetY());
        }
        // Z
        tmp = std::abs(rotationDelta.GetZ());
        if (maxAbsValue < tmp)
        {
            maxAbsValue = tmp;
            maxAxis = 2;
            isPositive = (0.f < rotationDelta.GetZ());
        }

        auto offset = (isPositive) ? 0 : 3;
        offset += maxAxis;
        m_Lifo.Append(
            MotionState({
                m_SamplingNumber,
                rawState.samplingNumber,
                static_cast<MotionType>(MotionType_RotatePlusX + offset)
            })
        );

        // NN_LOG("%s: Rotation finished\n", __FUNCTION__);
        // NN_LOG("RotationDelta : %f, %f, %f\n"
        //     , rotationDelta.GetX()
        //     , rotationDelta.GetY()
        //     , rotationDelta.GetZ()
        // );
        m_SamplingNumber++;

    }
    else if (m_IsRotationStarted)
    {
        // 回転中
        // NN_LOG("%f, %f, %f\n"
        //     , m_RotationState.rotation.GetX()
        //     , m_RotationState.rotation.GetY()
        //     , m_RotationState.rotation.GetZ()
        // );
    }

    // rotation の計算
    ::nn::util::Vector3f workVector;
    ::nn::util::VectorMultiply(&workVector,
                               rawState.gyro,
                               m_SamplingIntervalSeconds * AngularVelocityConvertUnitCoef);
    ::nn::util::VectorAdd(&m_RotationState.rotation,
                          m_RotationState.rotation,
                          workVector);
    // NN_LOG("%f, %f, %f\n"
    //     , m_RotationState.rotation.GetX()
    //     , m_RotationState.rotation.GetY()
    //     , m_RotationState.rotation.GetZ()
    // );
    m_RotationState.samplingNumber = m_RotationStateSamplingNumber;
    m_RotationStateLifo.Append(m_RotationState);

    m_RotationStateSamplingNumber++;
}

int RotateMotionDetector::GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT
{
    return m_Lifo.Read(pOutStates, count);
}
