﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
    @brief
    Npadを使ってボタン入力を取得するためのサンプルプログラム
*/

/**
    @page PageSampleHidNpadPalmaSimple Palma 利用時の Npad を使ったボタン入力の取得
    @tableofcontents

    @brief
    Npad を使って Palma を含めたボタン入力を取得するためのサンプルプログラムの解説です。
*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/oe.h>


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "BleUtility.h"
#include "ExecuteFunction.h"
#include "PalmaUtility.h"
#include "ApplicationLayout.h"

#include "Framework/Applet.h"
#include "Framework/ApplicationHeap.h"
#include "Framework/CursolControl.h"
#include "Framework/GraphicsSystem.h"
#include "Framework/Gesture.h"
#include "Framework/FontSystem.h"
#include "Framework/IrSensor.h"
#include "Framework/LocalCommunication.h"
#include "Framework/Npad.h"
#include "Framework/NpadAssignControl.h"

namespace
{
    util::PalmaController PalmaControl[frm::NpadIdNum];
    frm::NpadState  NpadState[frm::NpadIdNum];
    frm::SixAxis    SixAxis[frm::NpadIdNum];
    frm::CursolPosition Cursol;
    frm::GestureState GestureState;

    int FunctionMenu = 0;
    bool quit = true;

}//namespace

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}
void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}
void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

void Quit(frm::NpadState * state)
{
    for (auto index = 0; index < frm::NpadIdNum; index++)
    {
        if(state[index].press.Test<nn::hid::NpadButton::Plus>() && state[index].press.Test<nn::hid::NpadButton::Minus>())
        {
            quit = false;
        }
    }
}

void SelectFunctionMenu()
{
    frm::SwipeState swipe = gGesture.GetSwipe();

    if (swipe.IsSwipe == true)
    {
        if (swipe.Direction == nn::hid::GestureDirection_Left && swipe.x >= 750)
        {
            if (swipe.y >= 30 && swipe.y <= 230)
            {
                FunctionMenu = 2;     // アプレット選択
            }
            else if (swipe.y >= 240 && swipe.y <= 540)
            {
                FunctionMenu = 1;     // NpadAssign変更選択
            }
            else if (swipe.y >= 550 && swipe.y <= 700)
            {
                FunctionMenu = 3;     // Local通信選択
            }

        }
        else if (swipe.Direction == nn::hid::GestureDirection_Right && swipe.x >= 750)
        {
            FunctionMenu = 0;     // 通常画面
        }
    }
}

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

//------------------------------------------
// メモリアロケート
//------------------------------------------
nn::mem::StandardAllocator allocator;
nn::Bit8* pApplicationMemory = new nn::Bit8[ApplicationHeapSize];
allocator.Initialize(pApplicationMemory, ApplicationHeapSize);

//------------------------------------------
// Graphicsの生成とイニシャライズ
//------------------------------------------
GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
pGraphicsSystem->Initialize(&allocator, 1280, 720);

//------------------------------------------
// Fontの生成とイニシャライズ
//------------------------------------------
FontSystem* pFontSystem = new FontSystem();
pFontSystem->Initialize(&allocator, pGraphicsSystem);
nn::gfx::util::DebugFontTextWriter* pTextWriter = &pFontSystem->GetDebugFontTextWriter();

//------------------------------------------
// 初期化処理
//------------------------------------------
nn::oe::Initialize();
nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);     // BG時は動作を継続する

gApplet.Initialize();
gNpad.Initialize();
gCursol.Initialize();
gBle.Initialize();
gFs.Initialize();
gGesture.Initialize();

gFs.LoadFile(gPalma.m_ActivityFile.wave, &gPalma.m_ActivityFile.waveCount, "/Wave/");
gFs.LoadFile(gPalma.m_ActivityFile.rgbLed, &gPalma.m_ActivityFile.rgbLedCount, "/RgbLed/");
gFs.LoadFile(gPalma.m_Appdata.data, &gPalma.m_Appdata.count, "/ApplicationData/");

// 一発読み捨て
gNpad.Update();

//------------------------------------------
// MainLoop
//------------------------------------------
while (quit)
{
    gNpad.Update();
    gPalma.Update();

    for (auto index = 0; index < frm::NpadIdNum; index++)
    {
        NpadState[index] = gNpad.GetFromIndex(index);

        if (NpadState[index].style.Test<nn::hid::NpadStylePalma>())
        {
            gPalma.Enable(index);
        }

        if (NpadState[index].isConnected)
        {
            gIrSensor.Initialize(index);
        }
        else
        {
            gIrSensor.Finalize(index);
        }

        PalmaControl[index] = gPalma.GetController(index);
        SixAxis[index] = gNpad.GetSixAxisFromIndex(index);
    }

    gGesture.UpdateState(&GestureState);



    SelectFunctionMenu();

    if (FunctionMenu == 1)
    {
        NpadAssignmentControl(NpadState);
    }
    else if (FunctionMenu == 2)
    {
        gApplet.Control(NpadState);
    }
    else if (FunctionMenu == 3)
    {
        gLdn.Control(NpadState);
    }
    else
    {
        // カーソルの取得、更新
        gCursol.Update(NpadState, GestureState, gPalma.m_ActivityFile.waveCount, gPalma.m_ActivityFile.rgbLedCount, gPalma.m_Appdata.count);
        Cursol = gCursol.GetCursol();
        // カーソル選択で選ばれたメニュー内容の実行処理
        gExcuteFunction.Do(Cursol);
    }

    //-------------------------------------
    // 描画処理-Start-
    //-------------------------------------
    pGraphicsSystem->BeginDraw();

    gLayout.DrawDisplay(pGraphicsSystem, pTextWriter, NpadState, SixAxis, PalmaControl, &gBle.m_bleState, Cursol);

    pFontSystem->Draw();

    DrawNpadAssignMenu(pGraphicsSystem, pTextWriter, FunctionMenu);
    gApplet.DrawControl(pGraphicsSystem, pTextWriter, FunctionMenu);
    gLdn.Draw(pGraphicsSystem, pTextWriter, FunctionMenu);

    pFontSystem->Draw();

    //-------------------------------------
    // 描画処理-End-
    //-------------------------------------
    pGraphicsSystem->EndDraw();
    pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / 60));

    Quit(NpadState);

    /*::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));*/
}

    //-------------------------------------
    // 終了処理
    //-------------------------------------
    gPalma.StopScan();

    for (auto index = 0; index < frm::NpadIdNum; index++)
    {
        // SystemEvent の破棄
        if (PalmaControl[index].IsEnable)
        {
            nn::os::DestroySystemEvent(&PalmaControl[index].OperationEvent);
        }
    }
    gFs.Unmount();
    gBle.finalize();

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    allocator.Free(pApplicationMemory);
    allocator.Finalize();

    NN_ABORT();
}
