﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/os/os_Tick.h>

#include "RingBuffer.h"

const int PlrSamplingCountMax = 800;
const int RxPacketHistorySamplingCountMax = 800;
const auto PlrUpdateInterval = nn::TimeSpan::FromMilliSeconds(1500);  // msec

//!< Plr を定義した構造体
struct PlrInfo
{
    float plr;       //!< PLR
    int   received;  //!< 受信パケット数
};

//!< PLRを蓄積する FIFO
class PlrFifo : public RingBuffer<PlrInfo, PlrSamplingCountMax>
{
private:
    PlrInfo m_Max;
    PlrInfo m_Min;
    bool m_IsMaxMinSet;

public:
    void Initialize() NN_NOEXCEPT
    {
        this->Clear();
    }

    void AddSample(const float& plr, const int& received) NN_NOEXCEPT
    {
        PlrInfo input = { plr, received };
        this->push_back(input);
        if (m_IsMaxMinSet == false)
        {
            m_Max = input;
            m_Min = input;
            m_IsMaxMinSet = true;
        }

        m_Max.plr = std::max(m_Max.plr, plr);
        m_Min.plr = std::min(m_Max.plr, plr);
        m_Max.received = std::max(m_Max.received, received);
        m_Min.received = std::min(m_Max.received, received);
    }

    void Clear() NN_NOEXCEPT
    {
        m_Max.plr = 0;
        m_Min.plr = 0;
        m_IsMaxMinSet = false;
        this->clear();
    }

    PlrInfo GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

    PlrInfo GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }
};

//!< パケット履歴を蓄積する FIFO
class RxPacketHistoryFifo : public RingBuffer<nn::hid::detail::PacketInfo, PlrSamplingCountMax>
{
private:
    PlrFifo m_PlrFifo;
    uint64_t m_LastSamplingCount;
    nn::os::Tick m_LastPlrUpdate;
    int m_PacketReceived;
    int m_PacketDropped;
    bool m_OnPacketReceive;
    int m_ContinuousPacketDrop;
    int m_MaxContinuousPacketDrop;

public:
    void Initialize() NN_NOEXCEPT
    {
        this->Clear();
    }

    void AddSample(const nn::hid::detail::RxPacketHistory& input) NN_NOEXCEPT
    {
        int packetCount = std::min(static_cast<uint64_t>(nn::hid::detail::PacketHistoryCount), input.samplingCount - m_LastSamplingCount);
        for (int i = packetCount - 1; i >= 0; --i)
        {
            this->push_back(input.history[i]);

            if (input.history[i].status == nn::hid::detail::PacketStatus_Received)
            {
                ++m_PacketReceived;
                m_OnPacketReceive = true;
            }
            else
            {
                ++m_PacketDropped;
                if (m_OnPacketReceive)
                {
                    m_ContinuousPacketDrop = 1;
                    m_OnPacketReceive = false;
                }
                else
                {
                    ++m_ContinuousPacketDrop;
                    m_MaxContinuousPacketDrop = std::max(m_ContinuousPacketDrop, m_MaxContinuousPacketDrop);
                }
            }

            auto currentTick = nn::os::GetSystemTick();
            if ((currentTick - m_LastPlrUpdate).ToTimeSpan() > PlrUpdateInterval)
            {
                m_PlrFifo.AddSample(static_cast<float>(m_PacketDropped) / (m_PacketReceived + m_PacketDropped) * 100, m_PacketReceived);
                m_LastPlrUpdate = currentTick;
                m_PacketDropped = 0;
                m_PacketReceived = 0;
            }
        }
        m_LastSamplingCount = input.samplingCount;
    }

    void Clear() NN_NOEXCEPT
    {
        m_LastSamplingCount = 0;
        m_PacketReceived = 0;
        m_PacketDropped = 0;
        m_ContinuousPacketDrop = 0;
        m_MaxContinuousPacketDrop = 0;
        m_PlrFifo.Clear();
        this->clear();
    }

    PlrFifo& GetPlr() NN_NOEXCEPT
    {
        return m_PlrFifo;
    }

    int GetContinuouPacketDrop() NN_NOEXCEPT
    {
        return m_ContinuousPacketDrop;
    }

    int GetMaxContinuouPacketDrop() NN_NOEXCEPT
    {
        return m_MaxContinuousPacketDrop;
    }
};

