﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadCommonTypes.h>

// 押し時間検出をするボタン
const nn::hid::NpadButtonSet PressCountSupportedButtons[] = {
nn::hid::NpadButton::Plus::Mask,
nn::hid::NpadButton::Minus::Mask,
nn::hid::NpadButton::Up::Mask,
nn::hid::NpadButton::Down::Mask,
nn::hid::NpadButton::Left::Mask,
nn::hid::NpadButton::Right::Mask,
nn::hid::NpadButton::A::Mask,
nn::hid::NpadButton::B::Mask,
nn::hid::NpadButton::X::Mask,
nn::hid::NpadButton::Y::Mask,
nn::hid::NpadButton::StickLUp::Mask,
nn::hid::NpadButton::StickLDown::Mask
};
const int PressCountSupportedButtonsCount = NN_ARRAY_SIZE(PressCountSupportedButtons);
const int RepeatCountDefault = 24;
const int RepeatCountStep = 8;
const int RepeatCountMin = 16;
const int64_t LongThreshold = 24;

// 定数
struct PackedButton
{
    nn::hid::NpadButtonSet trigger;
    nn::hid::NpadButtonSet hold;
    nn::hid::NpadButtonSet repeat;
    nn::hid::NpadButtonSet longTrigger;
    nn::hid::NpadButtonSet longHold;

    int64_t pressCount[PressCountSupportedButtonsCount];
    int64_t repeatThreshold[PressCountSupportedButtonsCount];
    int64_t repeatGap[PressCountSupportedButtonsCount];

    void Reset()
    {
        trigger.Reset();
        hold.Reset();
        repeat.Reset();
        longTrigger.Reset();
        longHold.Reset();
    }

    void Update(const nn::hid::NpadButtonSet& button)
    {
        auto oldButtons = hold;
        auto oldLong = longHold;
        hold = button;

        // トリガの更新
        trigger = (hold ^ oldButtons & hold);
        repeat.Reset();
        longHold.Reset();

        // Repeat / LongPress の検出
        for (int i = 0; i < PressCountSupportedButtonsCount; ++i)
        {
            if ((PressCountSupportedButtons[i] & hold).IsAnyOn())
            {
                ++pressCount[i];
                if (pressCount[i] > repeatThreshold[i])
                {
                    repeat |= PressCountSupportedButtons[i];
                    if (repeatGap[i] >= RepeatCountMin)
                    {
                        repeatGap[i] -= RepeatCountStep;
                    }
                    repeatThreshold[i] += repeatGap[i];
                }
                if (pressCount[i] > LongThreshold)
                {
                    longHold |= PressCountSupportedButtons[i];
                }
            }
            else
            {
                pressCount[i] = 0;
                repeatGap[i] = RepeatCountDefault;
                repeatThreshold[i] = repeatGap[i];
            }
        }

        // 長押しトリガの更新
        longTrigger = (longHold ^ oldLong & longHold);
    };

    friend PackedButton operator & (const PackedButton& left, const nn::hid::NpadButtonSet& right) NN_NOEXCEPT
    {
        auto value = left;
        value.trigger &= right;
        value.hold &= right;
        value.repeat &= right;
        value.longTrigger &= right;
        value.longHold &= right;
        return value;
    };

    PackedButton& operator|=(const PackedButton& other) NN_NOEXCEPT
    {
        this->trigger |= other.trigger;
        this->hold |= other.hold;
        this->repeat |= other.repeat;
        this->longTrigger |= other.longTrigger;
        this->longHold |= other.longHold;
        return *this;
    }
};
