﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_UsbFullKey.h>
#include <nn/hid/system/hid_Npad.h>

struct AbstractedNpadState
{
    int64_t samplingNumber;                      //!< Npad の入力状態が更新される度に増加する値です。
    nn::hid::NpadButtonSet buttons;              //!< Npad のデジタルボタンの状態です。
    nn::hid::AnalogStickState analogStickL;      //!< Npad の左アナログスティックの状態です。
    nn::hid::AnalogStickState analogStickR;      //!< Npad の右アナログスティックの状態です。
    nn::hid::NpadAttributesSet attributes;       //!< Npad の状態を表す属性値です。
};

template <typename TState>
int GetAbstractedNpadStates(AbstractedNpadState* pOutState, int size, const nn::hid::NpadIdType& id)
{
    NN_SDK_REQUIRES_LESS_EQUAL(size, nn::hid::NpadStateCountMax);
    TState npadStates[nn::hid::NpadStateCountMax];

    auto gotCount = nn::hid::GetNpadStates(npadStates, size, id);
    NN_ABORT_UNLESS_GREATER(gotCount, 0);
    for (int i = 0; i < gotCount; ++i)
    {
        pOutState[i].buttons        = npadStates[i].buttons;
        pOutState[i].analogStickL   = npadStates[i].analogStickL;
        pOutState[i].analogStickR   = npadStates[i].analogStickR;
        pOutState[i].attributes     = npadStates[i].attributes;
        pOutState[i].samplingNumber = npadStates[i].samplingNumber;
    }

    return gotCount;
}

template <typename TState>
int GetAbstractedNpadSystemStates(AbstractedNpadState* pOutState, int size, const nn::hid::NpadIdType& id)
{
    NN_SDK_REQUIRES_LESS_EQUAL(size, nn::hid::NpadStateCountMax);
    TState npadStates[nn::hid::NpadStateCountMax];

    auto gotCount = nn::hid::system::GetNpadStates(npadStates, size, id);
    NN_ABORT_UNLESS_GREATER(gotCount, 0);
    for (int i = 0; i < gotCount; ++i)
    {
        pOutState[i].buttons        = npadStates[i].buttons;
        pOutState[i].analogStickL   = npadStates[i].analogStickL;
        pOutState[i].analogStickR   = npadStates[i].analogStickR;
        pOutState[i].attributes     = npadStates[i].attributes;
        pOutState[i].samplingNumber = npadStates[i].samplingNumber;
    }

    return gotCount;
}
