﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_ControllerSupport.h>

#include "NpadIdDefine.h"
#include "LogPrint.h"
#include "MenuControllerSupport.h"

class IMenuItemControllerSupport : public MenuItem
{
public:
    explicit IMenuItemControllerSupport(const std::string& name) NN_NOEXCEPT : MenuItem(name, true) {};

    void UpdateArg(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT
    {
        UpdateArgImpl(pValue);
    }

protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT = 0;
};

// 数の設定
class MenuItemPlayerCount : public IMenuItemControllerSupport
{
private:
    const int m_DefaultValue;

public:
    MenuItemPlayerCount(const std::string& name, const int& defaultValue) NN_NOEXCEPT :
        IMenuItemControllerSupport(name),
        m_DefaultValue(defaultValue)
    {
        AddSelectable("0");
        AddSelectable("1");
        AddSelectable("2");
        AddSelectable("3");
        AddSelectable("4");

        Reset();
    }

    void Reset() NN_NOEXCEPT
    {
        m_Value = m_DefaultValue;
    }
};

class MenuItemMinPlayer : public MenuItemPlayerCount
{
public:
    MenuItemMinPlayer() NN_NOEXCEPT : MenuItemPlayerCount("Min Player", 1) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->playerCountMin = m_Value;
    }
};

class MenuItemMaxPlayer : public MenuItemPlayerCount
{
public:
    MenuItemMaxPlayer() NN_NOEXCEPT : MenuItemPlayerCount("Max Player", 4) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->playerCountMax = m_Value;
    }
};

// 有効/無効設定
class MenuItemBooleanForControllerSupport : public IMenuItemControllerSupport
{
private:
    const bool m_DefaultValue;

public:
    MenuItemBooleanForControllerSupport(const std::string& name, bool defaultValue) NN_NOEXCEPT :
        IMenuItemControllerSupport(name),
        m_DefaultValue(defaultValue)
    {
        this->AddSelectable("\uE14B");
        this->AddSelectable("\uE14C");
        Reset();
    }

    void Reset() NN_NOEXCEPT
    {
        m_Value = m_DefaultValue ? 0 : 1;
    }

protected:
    bool IsEnabled() NN_NOEXCEPT
    {
        return (m_Value == 0);
    }
};

class MenuItemTakeOverCnct : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemTakeOverCnct() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("TakeOverCnct", true) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enableTakeOverConnection = this->IsEnabled();
    }
};

class MenuItemLeftJustify : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemLeftJustify() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("LeftJustify", true) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enableLeftJustify = this->IsEnabled();
    }
};

class MenuItemPermitJoyDual : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemPermitJoyDual() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("PermitJoyDual", true) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enablePermitJoyDual = this->IsEnabled();
    }
};

class MenuItemSingleMode : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemSingleMode() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("SingleMode", false) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enableSingleMode = this->IsEnabled();
    }
};

class MenuItemUsePlayerColor : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemUsePlayerColor() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("UsePlayerColor", true) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enableIdentificationColor = this->IsEnabled();
    }
};

class MenuItemUseExplainText : public MenuItemBooleanForControllerSupport
{
public:
    MenuItemUseExplainText() NN_NOEXCEPT : MenuItemBooleanForControllerSupport("UseExplainText", true) { };
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        pValue->enableExplainText = this->IsEnabled();
    }
};

// 色の設定
class MenuItemPlayerColor : public IMenuItemControllerSupport
{
private:
    const int m_PlayerIndex;

public:
    MenuItemPlayerColor(const std::string& name, const int& playerIndex) NN_NOEXCEPT :
        IMenuItemControllerSupport(name),
        m_PlayerIndex(playerIndex)
    {
        NN_SDK_REQUIRES_LESS(playerIndex, 4);
        AddSelectable("Black");
        AddSelectable("Gray");
        AddSelectable("White");
        AddSelectable("Red");
        AddSelectable("Green");
        AddSelectable("Blue");
        AddSelectable("Yellow");
        AddSelectable("Magenta");
        AddSelectable("Cyan");
        Reset();
    }

    void Reset() NN_NOEXCEPT
    {
        m_Value = 0;
    }
protected:

    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        auto pTarget = &pValue->identificationColor[m_PlayerIndex];
        auto nameString = m_Selectables.at(m_Value);
        if (nameString == "Black")
        {
            *pTarget = { {0, 0, 0, 255 } };
        }
        else if (nameString == "Gray")
        {
            *pTarget = { { 0x50, 0x50, 0x50, 0x50} };
        }
        else if (nameString == "White")
        {
            *pTarget = { {255, 255, 255, 255 } };
        }
        else if (nameString == "Red")
        {
            *pTarget = { {255, 0, 0, 255 } };
        }
        else if (nameString == "Green")
        {
            *pTarget = { {0, 255, 0, 255 } };
        }
        else if (nameString == "Blue")
        {
            *pTarget = { {0, 0, 255, 255 } };
        }
        else if (nameString == "Yellow")
        {
            *pTarget = { {255, 255, 0, 255 } };
        }
        else if (nameString == "Magenta")
        {
            *pTarget = { {255, 0, 255, 255 } };
        }
        else if (nameString == "Cyan")
        {
            *pTarget = { {0, 255, 255, 255 } };
        }
        else
        {
            *pTarget = { { 0x50, 0x50, 0x50, 0x50} };
        }
    }
};

// 説明用のテキスト
class MenuItemExplainText : public IMenuItemControllerSupport
{
private:
    const nn::hid::NpadIdType m_Id;

public:
    MenuItemExplainText(const std::string& name, const nn::hid::NpadIdType& id) NN_NOEXCEPT :
        IMenuItemControllerSupport(name),
        m_Id(id)
    {
        AddSelectable("Mario");
        AddSelectable("Japanese32");
        AddSelectable("alphabet32");
        AddSelectable("alphabet50");
        AddSelectable("iiii32");
        AddSelectable("www32");
        AddSelectable("None");
        AddSelectable("Luigi");
        AddSelectable("Peach");
        AddSelectable("Kinopio");
        Reset();
    }

    void Reset() NN_NOEXCEPT
    {
        m_Value = 0;
    }
protected:
    virtual void UpdateArgImpl(nn::hid::ControllerSupportArg* pValue) NN_NOEXCEPT NN_OVERRIDE
    {
        auto nameString = m_Selectables.at(m_Value);
        if (nameString == "Mario")
        {
            nn::hid::SetExplainText(pValue, "マリオ", m_Id);
        }
        else if (nameString == "Japanese32")
        {
            nn::hid::SetExplainText(pValue, "あいうえおかきくけこさしすせそたちつてとなにぬねのはひふへほまや", m_Id);
        }
        else if (nameString == "alphabet32")
        {
            nn::hid::SetExplainText(pValue, "abcdefghijklmnopqrstuvwxyzabcdef", m_Id);
        }
        else if (nameString == "alphabet50")
        {
            nn::hid::SetExplainText(pValue, "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwx", m_Id);
        }
        else if (nameString == "iiii32")
        {
            nn::hid::SetExplainText(pValue, "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii", m_Id);
        }
        else if (nameString == "www32")
        {
            nn::hid::SetExplainText(pValue, "wwwwwwwwwwwwwwwwwwwwwwwwwwwwwwww", m_Id);
        }
        else if (nameString == "None")
        {
            nn::hid::SetExplainText(pValue, "", m_Id);
        }
        else if (nameString == "Luigi")
        {
            nn::hid::SetExplainText(pValue, "ルイージ", m_Id);
        }
        else if (nameString == "Peach")
        {
            nn::hid::SetExplainText(pValue, "ピーチ", m_Id);
        }
        else if (nameString == "Kinopio")
        {
            nn::hid::SetExplainText(pValue, "キノピオ", m_Id);
        }
        else
        {
            nn::hid::SetExplainText(pValue, "クッパ", m_Id);
        }
    }
};

class MenuItemControllerSupportAssignment final : public MenuItemHolder
{
public:
    MenuItemControllerSupportAssignment() NN_NOEXCEPT :
        MenuItemHolder("Assignment", true)
    {
        AddMenuItem(new MenuItemMinPlayer());
        AddMenuItem(new MenuItemMaxPlayer());
        AddMenuItem(new MenuItemTakeOverCnct());
        AddMenuItem(new MenuItemLeftJustify());
        AddMenuItem(new MenuItemPermitJoyDual());
        AddMenuItem(new MenuItemSingleMode());
        AddMenuItem(new MenuItemUsePlayerColor());
        AddMenuItem(new MenuItemPlayerColor("PlayerColor1", 0));
        AddMenuItem(new MenuItemPlayerColor("PlayerColor2", 1));
        AddMenuItem(new MenuItemPlayerColor("PlayerColor3", 2));
        AddMenuItem(new MenuItemPlayerColor("PlayerColor4", 3));
        AddMenuItem(new MenuItemUseExplainText());
        AddMenuItem(new MenuItemExplainText("ExplainText1", nn::hid::NpadId::No1));
        AddMenuItem(new MenuItemExplainText("ExplainText2", nn::hid::NpadId::No2));
        AddMenuItem(new MenuItemExplainText("ExplainText3", nn::hid::NpadId::No3));
        AddMenuItem(new MenuItemExplainText("ExplainText4", nn::hid::NpadId::No4));
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::ControllerSupportArg arg;
        nn::hid::ControllerSupportResultInfo resultInfo;
        arg.SetDefault();
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            reinterpret_cast<IMenuItemControllerSupport*>(*it)->UpdateArg(&arg);
        }
        auto result = nn::hid::ShowControllerSupport(&resultInfo, arg);
        if (result.IsFailure())
        {
            HID_TOOL_WARN("CS_APPLET: Opening Assignment Mode Failed %03d-%04d", result.GetModule(), result.GetDescription());
        }
        else
        {
            HID_TOOL_LOG("CS_APPLET: Assignment Mode succeeded. NumOfPlayer: %d   SelectedId: %s\n", resultInfo.playerCount, GetNpadIdString(resultInfo.selectedId).c_str());
        }
    }
};

class MenuItemControllerSupportStrap final : public MenuItemHolder
{
public:
    MenuItemControllerSupportStrap() NN_NOEXCEPT :
        MenuItemHolder("Strap", true)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::ShowControllerStrapGuide();
    }
};

class MenuItemFirmwareUpdateForce final : public MenuItemBoolean
{
public:
    MenuItemFirmwareUpdateForce() NN_NOEXCEPT :
        MenuItemBoolean("ForceUpdate")
    {
        // 何もしない
    }
    bool IsForced() NN_NOEXCEPT
    {
        return this->IsEnabled();
    }
};

class MenuItemControllerSupportFirmwareUpdate final : public MenuItemHolder
{
public:
    MenuItemControllerSupportFirmwareUpdate() NN_NOEXCEPT :
        MenuItemHolder("FirmwareUpdate", true)
    {
        AddMenuItem(new MenuItemFirmwareUpdateForce());
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::ControllerFirmwareUpdateArg arg;
        arg.enableForceUpdate = reinterpret_cast<MenuItemFirmwareUpdateForce*>(m_pMenuItems.at(0))->IsForced();
        auto result = nn::hid::ShowControllerFirmwareUpdate(arg);
        if (result.IsFailure())
        {
            HID_TOOL_WARN("CS_APPLET: Firmware Update Failed %03d-%04d", result.GetModule(), result.GetDescription());
        }
    }
};

MenuItemControllerSupport::MenuItemControllerSupport() NN_NOEXCEPT :
    MenuItemHolder("Controller Support", true, true)
{
    AddMenuItem(new MenuItemControllerSupportAssignment());
    AddMenuItem(new MenuItemControllerSupportStrap());
    AddMenuItem(new MenuItemControllerSupportFirmwareUpdate());
}
