﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>

#include "MenuAssignment.h"

// デフォルト設定での 1本割り当て
class MenuItemAssignmentJoySingleDefault final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentJoySingleDefault() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Single-Default", true, 1)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetId;
        auto count = this->GetValue(&targetId, 1);
        NN_ABORT_UNLESS_EQUAL(count, 1);
        nn::hid::SetNpadJoyAssignmentModeSingle(targetId);
    }
};

// 左コン残しの 1本割り当て
class MenuItemAssignmentJoySingleKeepLeft final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentJoySingleKeepLeft() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Single-KeepLeft", true, 1)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetId;
        auto count = this->GetValue(&targetId, 1);
        NN_ABORT_UNLESS_EQUAL(count, 1);
        nn::hid::SetNpadJoyAssignmentModeSingle(targetId, nn::hid::NpadJoyDeviceType_Left);
    }
};

// 右コン残しの 1本割り当て
class MenuItemAssignmentJoySingleKeepRight final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentJoySingleKeepRight() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Single-KeepRight", true, 1)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetId;
        auto count = this->GetValue(&targetId, 1);
        NN_ABORT_UNLESS_EQUAL(count, 1);
        nn::hid::SetNpadJoyAssignmentModeSingle(targetId, nn::hid::NpadJoyDeviceType_Right);
    }
};

// 2本割り当て変更
class MenuItemAssignmentJoyDual final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentJoyDual() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Dual", true, 1)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetId;
        auto count = this->GetValue(&targetId, 1);
        NN_ABORT_UNLESS_EQUAL(count, 1);
        nn::hid::SetNpadJoyAssignmentModeDual(targetId);
    }
};

// マージ
class MenuItemAssignmentMerge final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentMerge() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Merge", true, 2)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetIds[2];
        auto count = this->GetValue(targetIds, NN_ARRAY_SIZE(targetIds));
        NN_ABORT_UNLESS_EQUAL(count, NN_ARRAY_SIZE(targetIds));
        nn::hid::MergeSingleJoyAsDualJoy(targetIds[0], targetIds[1]);
    }
};

// スワップ
class MenuItemAssignmentSwap final : public MenuItemNpadIdsSelectable
{
public:
    MenuItemAssignmentSwap() NN_NOEXCEPT :
        MenuItemNpadIdsSelectable("Swap", true, 2)
    {
        // 何もしない
    }
protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::NpadIdType targetIds[2];
        auto count = this->GetValue(targetIds, NN_ARRAY_SIZE(targetIds));
        NN_ABORT_UNLESS_EQUAL(count, NN_ARRAY_SIZE(targetIds));
        nn::hid::SwapNpadAssignment(targetIds[0], targetIds[1]);
    }
};

MenuItemAssignmentTop::MenuItemAssignmentTop() NN_NOEXCEPT :
    MenuItemHolder("Assignment", true, true)
{
    AddMenuItem(new MenuItemAssignmentJoySingleDefault());   // 1本持ち デフォルト
    AddMenuItem(new MenuItemAssignmentJoySingleKeepLeft());  // 1本持ち 左残し
    AddMenuItem(new MenuItemAssignmentJoySingleKeepRight()); // 1本持ち 右残し
    AddMenuItem(new MenuItemAssignmentJoyDual());            // 2本持ち
    AddMenuItem(new MenuItemAssignmentMerge());              // マージ
    AddMenuItem(new MenuItemAssignmentSwap());               // スワップ
}
