﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_UsbFullKey.h>
#include <nn/hid/hid_NpadDisableControlPadRestriction.h>
#include <nn/hid/system/hid_Handheld.h>
#include <nn/hid/system/hid_Npad.h>

#include "MenuAdditionalSettings.h"
#include "NpadStyleDefinition.h"

namespace {
    void SetBluetoothMode(bool enable) NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        nn::btm::system::EnableRadio(enable);
#else
        NN_UNUSED(enable);
        // 何もしない
#endif
    }

    bool GetBluetoothMode() NN_NOEXCEPT
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        return nn::btm::system::IsRadioEnabled();
#else
        return false;
#endif
    }
} // namespace

// 携帯機操作の有効無効
class MenuItemHandheldOnOff : public MenuItemBoolean
{
private:
    bool m_Enabled;

public:
    MenuItemHandheldOnOff() NN_NOEXCEPT :
        MenuItemBoolean("Handheld On/Off"),
        m_Enabled(true) // デフォルト無効
    {
        // 何もしない
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Enabled = this->IsEnabled();
        if(m_Enabled)
        {
            nn::hid::system::EnableHandheldHids();
        }
        else
        {
            nn::hid::system::DisableHandheldHids();
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        this->UpdateValue(m_Enabled);
    }
};

// Bluetooth の設定
class MenuItemBluetooth : public MenuItemBoolean
{
public:
    MenuItemBluetooth() NN_NOEXCEPT :
        MenuItemBoolean("Bluetooth")
    {
        // 何もしない
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        SetBluetoothMode(this->IsEnabled());
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        this->UpdateValue(GetBluetoothMode());
    }
};

// Pro Controller の USB 通信設定
class MenuItemUsbOnFullKey : public MenuItemBoolean
{
public:
    MenuItemUsbOnFullKey() NN_NOEXCEPT :
        MenuItemBoolean("Usb On FullKey")
    {
        // 何もしない
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::EnableUsbFullKeyController(this->IsEnabled());
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        this->UpdateValue(nn::hid::IsUsbFullKeyControllerEnabled());
    }
};


// スタイル毎の有効/無効設定
class MenuItemControlPad : public MenuItem
{
private:
    const nn::hid::NpadStyleSet m_Style;
    bool m_CurrentValue;

public:
    MenuItemControlPad(const std::string& name, const nn::hid::NpadStyleSet& style) NN_NOEXCEPT :
        MenuItem(name, true),
        m_Style(style)
    {
        this->AddSelectable("None");
        this->AddSelectable("Left/Up");

        // デフォルト設定
        if (m_Style.Test<nn::hid::NpadStyleJoyDual>() || m_Style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            m_CurrentValue = false;
        }
        else
        {
            m_CurrentValue = true;
        }
    }

    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_Value)
        {
        case 0:
            m_CurrentValue = false;
            break;
        case 1:
            m_CurrentValue = true;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        nn::hid::EnableControlPadRestriction(m_Style, m_CurrentValue);
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_CurrentValue)
        {
            m_Value = 1;
        }
        else
        {
            m_Value = 0;
        }
    }
};

// Npad の禁則処理に関する設定
class MenuItemNpadControlPadRestriction final : public MenuItemHolder
{
public:
    MenuItemNpadControlPadRestriction() NN_NOEXCEPT :
        MenuItemHolder("ControlPad Restriction", true)
    {
        for (auto& style : NpadStyles)
        {
            AddMenuItem(new MenuItemControlPad(style.styleString, style.style));
        }
    }

protected:
    virtual void Proceed() NN_NOEXCEPT NN_OVERRIDE
    {
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
             it != m_pMenuItems.end();
             ++it)
        {
            (*it)->Proceed();
        }
    }
};

MenuItemAdditionalSettings::MenuItemAdditionalSettings() NN_NOEXCEPT :
    MenuItemHolder("Additional Settings", true, true)
{
    AddMenuItem(new MenuItemHandheldOnOff());      // 携帯機操作 有効/無効設定
    AddMenuItem(new MenuItemBluetooth());          // Bluetooth の ON/OFF 設定
    AddMenuItem(new MenuItemUsbOnFullKey());       // Pro Controller の 有線USB通信機能 の ON/OFF 設定
    AddMenuItem(new MenuItemNpadControlPadRestriction()); // 禁則処理の設定
}

void MenuItemAdditionalSettings::Proceed() NN_NOEXCEPT
{
    for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
        it != m_pMenuItems.end();
        ++it)
    {
        (*it)->Proceed();
    }
}

