﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/init/init_Malloc.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_UsbFullKey.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_VibrationDeviceApi.h>
#include <nn/hid/hid_VibrationGcErm.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/system/hid_Joy.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_PlayReport.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/xcd/xcd_Sleep.h>

#include "NpadController.h"
#include "LogPrint.h"
#include "PlayReportUsageMonitor.h"

nn::os::SystemEventType g_NfcDeviceUpdateEvent;
nn::os::MultiWaitHolderType g_NfcDeviceUpdateHolder;
nn::os::SystemEventType g_NfcActivateEvent[NpadIdCountMax];
nn::os::MultiWaitHolderType g_NfcActivateHolder[NpadIdCountMax];
nn::os::SystemEventType g_IrSensorEvent[NpadIdCountMax];
nn::os::MultiWaitHolderType g_IrSensorHolder[NpadIdCountMax];
nn::os::SystemEventType g_JoyConDetachEvent;
nn::os::MultiWaitHolderType g_JoyConDetachHolder;
nn::os::SystemEventType g_PlayReportUsageUpdateEvent;
nn::os::MultiWaitHolderType g_PlayReportUsageUpdateHolder;
nn::os::SystemEventType g_PlayReportRegistrationUpdateEvent;
nn::os::MultiWaitHolderType g_PlayReportRegistrationUpdateHolder;
nn::os::SystemEventType g_DeviceRegisteredEvent;
nn::os::MultiWaitHolderType g_DeviceRegisteredHolder;
nn::os::MultiWaitType g_EventMonitoringMultiWait;
nn::os::ThreadType g_EventMonitoringThread;
NN_ALIGNAS(4096) char g_EventMonitoringThreadStack[4096];

void EventMonitoringThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    while (NN_STATIC_CONDITION(true))
    {
        // デバイス情報の更新を待機
        auto pHolder = nn::os::WaitAny(&g_EventMonitoringMultiWait);

        if (pHolder == &g_NfcDeviceUpdateHolder)
        {
            nn::os::ClearSystemEvent(&g_NfcDeviceUpdateEvent);
            HID_TOOL_LOG("EVENT: Nfc Device Update Event");
            nn::hid::NpadIdType nfcIds[NpadIdCountMax];

            auto nfcCount = nn::hid::system::GetNpadsWithNfc(nfcIds, NpadIdCountMax);
            for (int i = 0; i < nfcCount; i++)
            {
                ::nn::xcd::DeviceHandle handle;
                auto nfcResult = nn::hid::system::GetXcdHandleForNpadWithNfc(&handle, nfcIds[i]);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nfcResult);
                HID_TOOL_LOG("    %s   XcdDeviceHandle: %08x%08x", GetNpadIdString(nfcIds[i]).c_str(),  handle._upper, handle._lower);
            }
        }
        else if (pHolder == &g_JoyConDetachHolder)
        {
            nn::os::ClearSystemEvent(&g_JoyConDetachEvent);
            HID_TOOL_LOG("EVENT: JoyCon Detached during Bluetooth Off");
        }
        else if (pHolder == &g_PlayReportUsageUpdateHolder)
        {
            nn::os::ClearSystemEvent(&g_PlayReportUsageUpdateEvent);
            HID_TOOL_LOG("EVENT: PlayReport Usage Updated");
            PlayReportUsageMonitor::GetInstance().Update();
        }
        else if (pHolder == &g_PlayReportRegistrationUpdateHolder)
        {
            nn::os::ClearSystemEvent(&g_PlayReportRegistrationUpdateEvent);
            HID_TOOL_LOG("EVENT: Device Registration");
        }
        else if (pHolder == &g_DeviceRegisteredHolder)
        {
            nn::os::ClearSystemEvent(&g_DeviceRegisteredEvent);
            HID_TOOL_LOG("EVENT: Device Registraion on new device (For Controller Support)");
        }
        else
        {
            for (int i = 0; i < NpadIdCountMax; i++)
            {
                if (pHolder == &g_NfcActivateHolder[i])
                {
                    nn::os::ClearSystemEvent(&g_NfcActivateEvent[i]);
                    HID_TOOL_LOG("EVENT: %s    Nfc Activate Event", GetNpadIdString(NpadIds[i]).c_str());
                    break;
                }
            }
            for (int i = 0; i < NpadIdCountMax; i++)
            {
                if (pHolder == &g_IrSensorHolder[i])
                {
                    ::nn::xcd::DeviceHandle handle;
                    ::nn::Result irSensorResult;

                    nn::os::ClearSystemEvent(&g_IrSensorEvent[i]);
                    HID_TOOL_LOG("EVENT: %s    IrSensor Event", GetNpadIdString(NpadIds[i]).c_str());
                    switch (nn::hid::system::GetIrSensorState(NpadIds[i]))
                    {
                    case nn::hid::system::IrSensorState_Activated:
                    case nn::hid::system::IrSensorState_Deactivated:
                        irSensorResult = nn::hid::system::GetXcdHandleForNpadWithIrSensor(&handle, NpadIds[i]);
                        NN_ABORT_UNLESS_RESULT_SUCCESS(irSensorResult);
                        HID_TOOL_LOG("    XcdDeviceHandle: %08x%08x", handle._upper, handle._lower);
                        break;
                    case nn::hid::system::IrSensorState_NotConnected:
                    case nn::hid::system::IrSensorState_NotSupported:
                        HID_TOOL_LOG("    NoIrSensorAvailable");
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;                    }
                    break;
                }
            }
        }
    }
} // NOLINT(impl/function_size)

void InitializeEventMonitor(NpadControllerList& controllers) NN_NOEXCEPT
{
    nn::os::InitializeMultiWait(&g_EventMonitoringMultiWait);

    nn::hid::system::BindJoyDetachOnBluetoothOffEvent(&g_JoyConDetachEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_JoyConDetachHolder, &g_JoyConDetachEvent);
    nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_JoyConDetachHolder);

    nn::hid::system::BindNfcDeviceUpdateEvent(&g_NfcDeviceUpdateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_NfcDeviceUpdateHolder, &g_NfcDeviceUpdateEvent);
    nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_NfcDeviceUpdateHolder);

    nn::hid::system::BindPlayReportControllerUsageUpdateEvent(&g_PlayReportUsageUpdateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_PlayReportUsageUpdateHolder, &g_PlayReportUsageUpdateEvent);
    nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_PlayReportUsageUpdateHolder);

    nn::hid::system::BindPlayReportRegisteredDeviceUpdateEvent(&g_PlayReportRegistrationUpdateEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_PlayReportRegistrationUpdateHolder, &g_PlayReportRegistrationUpdateEvent);
    nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_PlayReportRegistrationUpdateHolder);

    nn::hid::system::BindDeviceRegisteredEventForControllerSupport(&g_DeviceRegisteredEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&g_DeviceRegisteredHolder, &g_DeviceRegisteredEvent);
    nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_DeviceRegisteredHolder);

    int index = 0;
    for (auto& controller : controllers)
    {
        //Npadの初期化
        controller.Initialize(NpadIds[index]);

        // NFC アクティブ状態の更新
        nn::hid::system::BindNfcActivateEvent(NpadIds[index], &g_NfcActivateEvent[index], nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&g_NfcActivateHolder[index], &g_NfcActivateEvent[index]);
        nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_NfcActivateHolder[index]);

        nn::hid::system::BindIrSensorEvent(NpadIds[index], &g_IrSensorEvent[index], nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&g_IrSensorHolder[index], &g_IrSensorEvent[index]);
        nn::os::LinkMultiWaitHolder(&g_EventMonitoringMultiWait, &g_IrSensorHolder[index]);

        ++index;
    }

    nn::os::CreateThread(&g_EventMonitoringThread, EventMonitoringThread, nullptr, g_EventMonitoringThreadStack, sizeof(g_EventMonitoringThreadStack), nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_EventMonitoringThread);
}
