﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/hid/hid_Vibration.h>

namespace {

    const ::nn::hid::VibrationValue VibrationValueArray[] = {
        ::nn::hid::VibrationValue::Make(1.0000f, 160.00f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.9333f, 160.00f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.8667f, 160.00f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.8000f, 160.00f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 113.14f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 134.54f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 190.27f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 226.27f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 160.00f, 1.0000f, 160.00f),
        ::nn::hid::VibrationValue::Make(0.5000f, 160.00f, 0.5000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.5000f, 160.00f, 0.1000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.9333f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.8667f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.8000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 226.27f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 269.09f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 380.55f),
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 452.55f)
    };

    void CheckVibrationValue(const nn::hid::VibrationValue* pSrc, const nn::hid::VibrationValue* pDst)
    {
        EXPECT_FLOAT_EQ(pSrc->amplitudeHigh, pDst->amplitudeHigh);
        EXPECT_FLOAT_EQ(pSrc->amplitudeLow,  pDst->amplitudeLow);
        EXPECT_FLOAT_EQ(pSrc->frequencyHigh, pDst->frequencyHigh);
        EXPECT_FLOAT_EQ(pSrc->frequencyLow,  pDst->frequencyLow);
    }

    void CheckVibrationModulation(const nn::hid::VibrationModulation* pSrc, const nn::hid::VibrationModulation* pDst)
    {
        EXPECT_FLOAT_EQ(pSrc->gainHigh, pDst->gainHigh);
        EXPECT_FLOAT_EQ(pSrc->gainLow, pDst->gainLow);
        EXPECT_FLOAT_EQ(pSrc->pitchHigh, pDst->pitchHigh);
        EXPECT_FLOAT_EQ(pSrc->pitchLow, pDst->pitchLow);
    }
}

//!< VibrationNodeConnectionに関するテスト（正常系）
TEST(VibrationNodeConnectionSuite, NormalTest1)
{
    nn::hid::VibrationWriter nodeSrc;
    nn::hid::VibrationMixer nodeDst;
    nn::hid::VibrationNodeConnection connection;

    connection.Disconnect();
    EXPECT_EQ(connection.IsConnected(), false);

    EXPECT_EQ(connection.Connect(&nodeSrc, &nodeDst), true);
    EXPECT_EQ(connection.IsConnected(), true);

    //GetSourceが正しく機能するか
    EXPECT_EQ(connection.GetSource(), &nodeSrc);

    //GetDestinationが正しく機能するか
    EXPECT_EQ(connection.GetDestination(), &nodeDst);

    //接続ノードは一致するか
    EXPECT_EQ(nodeSrc.GetConnectionTo(&nodeDst), &connection);

    //GetModulationが正しく機能するか
    nn::hid::VibrationModulation mod = connection.GetModulation();
    nn::hid::VibrationModulation zero = nn::hid::VibrationModulation::Make();
    CheckVibrationModulation(&mod, &zero);

    //Writerに入力した振動値が正しくMixerから取り出せるかの確認 x1編
    for (const auto& src : VibrationValueArray)
    {
        nodeSrc.Write(src);
        nn::hid::VibrationNode::Update();
        nn::hid::VibrationValue dst = nodeDst.GetCurrentVibration();
        CheckVibrationValue(&src, &dst);
    }

    //SetModulationが正しく機能するか
    nn::hid::VibrationModulation twice = nn::hid::VibrationModulation::Make(2.0f, 2.0f, 2.0f, 2.0f);
    connection.SetModulation(twice);
    mod = connection.GetModulation();
    CheckVibrationModulation(&mod, &twice);

    //Writerに入力した振動値が正しくMixerから取り出せるかの確認 x2編
    for (auto src : VibrationValueArray)
    {
        nodeSrc.Write(src);
        nn::hid::VibrationNode::Update();
        nn::hid::VibrationValue dst = nodeDst.GetCurrentVibration();

        twice.ApplyModulation(&src);
        CheckVibrationValue(&src, &dst);
    }

    connection.Disconnect();
    EXPECT_EQ(connection.IsConnected(), false);
}

//!< VibrationNodeConnectionに関するテスト（正常系）
TEST(VibrationNodeConnectionSuite, NormalTest2)
{
    nn::hid::VibrationWriter nodeSrc1;
    nn::hid::VibrationWriter nodeSrc2;
    nn::hid::VibrationMixer nodeDst1;
    nn::hid::VibrationMixer nodeDst2;
    nn::hid::VibrationNodeConnection connection1;
    nn::hid::VibrationNodeConnection connection2;

    //SrcとDstを接続
    EXPECT_EQ(connection1.Connect(&nodeSrc1, &nodeDst1), true);
    EXPECT_EQ(connection1.IsConnected(), true);

    //Disconnectしない限りConnectはできない
    EXPECT_EQ(connection1.Connect(&nodeSrc1, &nodeDst1), false);
    EXPECT_EQ(connection1.IsConnected(), true);
    EXPECT_EQ(connection1.Connect(&nodeSrc1, &nodeDst2), false);
    EXPECT_EQ(connection1.IsConnected(), true);
    EXPECT_EQ(connection1.Connect(&nodeSrc2, &nodeDst1), false);
    EXPECT_EQ(connection1.IsConnected(), true);
    EXPECT_EQ(connection1.Connect(&nodeSrc2, &nodeDst2), false);
    EXPECT_EQ(connection1.IsConnected(), true);

    //他のノードで同じ接続は作れない
    EXPECT_EQ(connection2.IsConnected(), false);
    EXPECT_EQ(connection2.Connect(&nodeSrc1, &nodeDst1), false);
    EXPECT_EQ(connection2.IsConnected(), false);
    EXPECT_EQ(connection2.Connect(&nodeSrc2, &nodeDst2), true);
    EXPECT_EQ(connection2.IsConnected(), true);

    //Discconetを複数回呼ぶ
    connection1.Disconnect();
    EXPECT_EQ(connection1.IsConnected(), false);
    connection1.Disconnect();
    EXPECT_EQ(connection1.IsConnected(), false);
    connection2.Disconnect();
    EXPECT_EQ(connection1.IsConnected(), false);
}

//!< VibrationNodeConnectionに関するテスト（正常系）
TEST(VibrationNodeSuite, NormalTest1)
{
    nn::hid::VibrationWriter nodeSrc1;
    nn::hid::VibrationMixer nodeDst1;
    nn::hid::VibrationMixer nodeDst2;

    EXPECT_EQ(nodeSrc1.GetConnectionTo(&nodeDst1), nullptr);
    nn::hid::VibrationNodeConnection connection(&nodeSrc1, &nodeDst1);

    EXPECT_NE(nodeSrc1.GetConnectionTo(&nodeDst1), nullptr);
    EXPECT_EQ(nodeSrc1.GetConnectionTo(&nodeDst1), &connection);
    EXPECT_EQ(nodeSrc1.IsConnectedTo(&nodeDst1), true);

    //接続していないものを削除
    nodeSrc1.DisconnectFrom(&nodeDst2);
    EXPECT_EQ(nodeSrc1.IsConnectedTo(&nodeDst1), true);

    //接続しているものを削除
    nodeSrc1.DisconnectFrom(&nodeDst1);
    EXPECT_EQ(nodeSrc1.IsConnectedTo(&nodeDst1), false);
    connection.Connect(&nodeSrc1, &nodeDst1);

    //接続しているものから取得
    nn::hid::VibrationModulation mod1 = nodeSrc1.GetModulationTo(&nodeDst1);
    nn::hid::VibrationModulation zero = nn::hid::VibrationModulation::Make();
    CheckVibrationModulation(&mod1, &zero);

    //接続していないものから取得
    nn::hid::VibrationModulation mod2 = nodeSrc1.GetModulationTo(&nodeDst2);
    CheckVibrationModulation(&mod2, &zero);

    //接続しているものに設定
    nodeSrc1.SetModulationTo(&nodeDst1, mod1);

    //接続していないものに設定
    nodeSrc1.SetModulationTo(&nodeDst2, mod2);
}

//振動値に対する調整係数に関するテスト
TEST(VibrationModulationSuite, NormalTest1)
{
    const ::nn::hid::VibrationValue VibrationValueArray2[] = {
        ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
        ::nn::hid::VibrationValue::Make(1.0000f, 160.00f, 0.5000f, 320.00f),
        ::nn::hid::VibrationValue::Make(0.5000f,  80.00f, 0.2500f, 160.00f),
    };

    const ::nn::hid::VibrationModulation VibrationModulationArray2[] = {
        ::nn::hid::VibrationModulation::Make(1.0f,1.0f,1.0f,1.0f),
        ::nn::hid::VibrationModulation::Make(0.5f,0.5f,0.5f,0.5f),
    };

    //Makeチェック
    EXPECT_FLOAT_EQ(VibrationModulationArray2[0].gainHigh, 1.0f);
    EXPECT_FLOAT_EQ(VibrationModulationArray2[0].gainLow, 1.0f);
    EXPECT_FLOAT_EQ(VibrationModulationArray2[0].pitchHigh, 1.0f);
    EXPECT_FLOAT_EQ(VibrationModulationArray2[0].pitchLow, 1.0f);

    //初期値チェック
    nn::hid::VibrationModulation mod = nn::hid::VibrationModulation::Make();
    CheckVibrationModulation(&mod, &VibrationModulationArray2[0]);

    //ApplyModulatio確認x1
    nn::hid::VibrationValue value = VibrationValueArray2[0];
    mod = VibrationModulationArray2[0];
    mod.ApplyModulation(&value);
    CheckVibrationValue(&value, &VibrationValueArray2[0]);

    //ApplyModulatio確認x0.5
    value = VibrationValueArray2[1];
    mod = VibrationModulationArray2[1];
    mod.ApplyModulation(&value);
    CheckVibrationValue(&value, &VibrationValueArray2[2]);

}

