﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/hid/hid_Vibration.h>

namespace {

    const ::nn::hid::VibrationMixMode MixModes[] = {
        ::nn::hid::VibrationMixMode_None,                      //!< ミックスを行わない (つねにデフォルトの振動値となる)
        ::nn::hid::VibrationMixMode_MaxAmplitude,              //!< 低周波帯と高周波帯の振幅の合計値が大きい方を採用する方式
        ::nn::hid::VibrationMixMode_MaxAmplitudePerSubband,    //!< 低周波帯と高周波帯それぞれで振幅が大きい方を採用する方式
        ::nn::hid::VibrationMixMode_AmplitudeSum               //!< 振幅値を加算する方式 (周波数は各入力を振幅で重み付けした加重平均値を採用)
    };

    const int MixModesMax = NN_ARRAY_SIZE(MixModes);

    const ::nn::hid::VibrationValue VibrationValueArray1[] = {
        ::nn::hid::VibrationValue::Make(0.5000f, 200.00f, 0.5000f, 300.00f),
    };

    const ::nn::hid::VibrationValue VibrationValueArray2[] = {
        ::nn::hid::VibrationValue::Make(0.2000f, 100.00f, 0.2000f, 200.00f),
        ::nn::hid::VibrationValue::Make(0.2000f, 100.00f, 0.8000f, 200.00f),
        ::nn::hid::VibrationValue::Make(0.5000f, 100.00f, 0.5000f, 200.00f),
        ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.2000f, 200.00f),
        ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.8000f, 200.00f),
    };

    const int TestSetsMax = NN_ARRAY_SIZE(VibrationValueArray2);

    const ::nn::hid::VibrationValue ExpectedVibrationValueArray[MixModesMax][TestSetsMax] = {
        {
            ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
            ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
            ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
            ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
            ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.0000f, 320.00f),
        },
        {
            ::nn::hid::VibrationValue::Make(0.5000f, 200.00f, 0.5000f, 300.00f),
            ::nn::hid::VibrationValue::Make(0.2000f, 100.00f, 0.8000f, 200.00f),
            ::nn::hid::VibrationValue::Make(0.5000f, 100.00f, 0.5000f, 200.00f),
            ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.2000f, 200.00f),
            ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.8000f, 200.00f),
        },
        {
            ::nn::hid::VibrationValue::Make(0.5000f, 200.00f, 0.5000f, 300.00f),
            ::nn::hid::VibrationValue::Make(0.5000f, 200.00f, 0.8000f, 200.00f),
            ::nn::hid::VibrationValue::Make(0.5000f, 100.00f, 0.5000f, 200.00f),
            ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.5000f, 300.00f),
            ::nn::hid::VibrationValue::Make(0.8000f, 100.00f, 0.8000f, 200.00f),
        },
        {
            ::nn::hid::VibrationValue::Make(0.7000f, 171.42f, 0.7000f, 271.43f),
            ::nn::hid::VibrationValue::Make(0.7000f, 171.42f, 1.3000f, 238.46f),
            ::nn::hid::VibrationValue::Make(1.0000f, 150.00f, 1.0000f, 250.00f),
            ::nn::hid::VibrationValue::Make(1.3000f, 138.46f, 0.7000f, 271.43f),
            ::nn::hid::VibrationValue::Make(1.3000f, 138.46f, 1.3000f, 238.46f),
        },
    };

    void CheckVibrationValue(const nn::hid::VibrationValue* pSrc, const nn::hid::VibrationValue* pDst)
    {
        ASSERT_NEAR(pSrc->amplitudeHigh, pDst->amplitudeHigh, 0.001);
        ASSERT_NEAR(pSrc->amplitudeLow,  pDst->amplitudeLow,  0.001);
        ASSERT_NEAR(pSrc->frequencyHigh, pDst->frequencyHigh, 0.01);
        ASSERT_NEAR(pSrc->frequencyLow,  pDst->frequencyLow,  0.01);
    }
}

//!< VibrationMixerに関するテスト（正常系）
TEST(VibrationMixerSuite, NormalTest1)
{
    ::nn::hid::VibrationWriter src1;
    ::nn::hid::VibrationWriter src2;
    ::nn::hid::VibrationMixer mix;
    ::nn::hid::VibrationNodeConnection connection1;
    ::nn::hid::VibrationNodeConnection connection2;

    EXPECT_EQ(connection1.Connect(&src1, &mix), true);
    EXPECT_EQ(connection2.Connect(&src2, &mix), true);

    //デフォルトのモード確認
    EXPECT_EQ(mix.GetMixMode() , ::nn::hid::VibrationMixMode_MaxAmplitudePerSubband);

    src1.Write(VibrationValueArray1[0]);

    //各Mixmodeに対して2つの入力に対する計算結果を照合する
    for (int i = 0; i < MixModesMax; i++)
    {
        mix.SetMixMode(MixModes[i]);
        EXPECT_EQ(mix.GetMixMode(), MixModes[i]);

        for (int j = 0; j < TestSetsMax; j++)
        {
            src2.Write(VibrationValueArray2[j]);
            ::nn::hid::VibrationNode::Update();
            ::nn::hid::VibrationValue value = mix.GetCurrentVibration();
            CheckVibrationValue(&value, &ExpectedVibrationValueArray[i][j]);
        }
    }
}

//!< VibrationMixerに関するテスト（正常系）
TEST(VibrationMixerSuite, NormalTest2)
{
    const int NodesMax = 100;
    ::nn::hid::VibrationMixer* pMixer;
    ::nn::hid::VibrationWriter* srcList[NodesMax];
    ::nn::hid::VibrationNodeConnection* connectionList[NodesMax];

    pMixer = new ::nn::hid::VibrationMixer();

    //ノードを100まで追加するテスト
    for (int i = 0; i < NodesMax; i++)
    {
        srcList[i] = new ::nn::hid::VibrationWriter();
        connectionList[i] = new ::nn::hid::VibrationNodeConnection();
        EXPECT_EQ(connectionList[i]->Connect(srcList[i], pMixer), true);
        ::nn::hid::VibrationNode::Update();
        pMixer->GetCurrentVibration();
    }

    //ノードを0まで削除するテスト
    for (int i = 0; i < NodesMax; i++)
    {
        connectionList[i]->Disconnect();
        EXPECT_EQ(connectionList[i]->IsConnected(), false);
        delete srcList[i];
        delete connectionList[i];
        ::nn::hid::VibrationNode::Update();
        pMixer->GetCurrentVibration();
    }

    delete pMixer;
}
