﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_DebugPad.h"

namespace
{

::nn::hid::DebugPadState g_States[::nn::hid::DebugPadStateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(DebugPadReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::DebugPadState& state : g_States)
    {
        state = ::nn::hid::DebugPadState();
        state.samplingNumber = 0xDEADBEEF;
        state.buttons.Set(1, true);
        state.analogStickR.x = 0x2222;
        state.analogStickR.y = 0x3333;
        state.analogStickL.x = 0x4444;
        state.analogStickL.y = 0x5555;
    }

    ::nn::hid::InitializeDebugPad();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetDebugPadStates(g_States,
                                             ::nn::hid::DebugPadStateCountMax);

    EXPECT_GE(::nn::hid::DebugPadStateCountMax, count);

    for (int i = count; i < ::nn::hid::DebugPadStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
        ::nn::hid::DebugPadButtonSet buttons;
        buttons.Reset();
        buttons.Set(1, true);
        EXPECT_EQ(buttons, g_States[i].buttons);
        EXPECT_EQ(0x2222, g_States[i].analogStickR.x);
        EXPECT_EQ(0x3333, g_States[i].analogStickR.y);
        EXPECT_EQ(0x4444, g_States[i].analogStickL.x);
        EXPECT_EQ(0x5555, g_States[i].analogStickL.y);
    }

    ::nn::hid::debug::FinalizeDebugPad();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(DebugPadReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(DebugPadReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::InitializeDebugPad();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count =
            ::nn::hid::GetDebugPadStates(g_States,
                                         ::nn::hid::DebugPadStateCountMax);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::DebugPadState& lhs = g_States[j];
            const ::nn::hid::DebugPadState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::DebugPadStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeDebugPad();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
