﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_DebugPad.h"

namespace
{

::nn::hid::DebugPadState g_States1[::nn::hid::DebugPadStateCountMax];
::nn::hid::DebugPadState g_States2[::nn::hid::DebugPadStateCountMax];

//!< DebugPad の自動操作状態をファジングします。
void FuzzDebugPadAutoPilotState(
    ::nn::hid::debug::DebugPadAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< DebugPad の入力状態が入力無し状態であることを期待します。
void ExpectDefaultDebugPadState(
    const ::nn::hid::DebugPadState& state) NN_NOEXCEPT;

//!< DebugPad の入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedDebugPadState(
    const ::nn::hid::DebugPadState& state) NN_NOEXCEPT;

//!< DebugPad の入力状態が同じドライバの状態から決定されたことを期待します。
void ExpectSameOriginDebugPadStates(
    const ::nn::hid::DebugPadState& lhs,
    const ::nn::hid::DebugPadState& rhs) NN_NOEXCEPT;

//!< DebugPad の入力状態が隣り合っていることを期待します。
void ExpectAdjacentDebugPadStates(
    const ::nn::hid::DebugPadState& lhs,
    const ::nn::hid::DebugPadState& rhs) NN_NOEXCEPT;

} // namespace

//!< DebugPad の初期化処理は自動操作状態を正しく管理するか
TEST(DebugPadAutoPilotSuite, InitializationTest1)
{
    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState;
    FuzzDebugPadAutoPilotState(&autoPilotState, 1);

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::GetDebugPadState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedDebugPadState(g_States1[0]);
    }

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::FinalizeDebugPad();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::GetDebugPadState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedDebugPadState(g_States1[0]);
    }

    ::nnt::hid::FinalizeDebugPadForAutoPilot();

    ::nn::hid::InitializeDebugPad();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    // DebugPad が完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::GetDebugPadState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultDebugPadState(g_States1[0]);
    }

    ::nnt::hid::FinalizeDebugPadForAutoPilot();
}

//!< DebugPad の入力状態は不正な自動操作状態が設定されても仕様を保証するか
TEST(DebugPadAutoPilotSuite, InvalidArgumentTest1)
{
    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState = {};

    // 未定義のフラグを有効化
    autoPilotState.attributes.Flip();

    // 禁則処理に違反
    autoPilotState.buttons.Set<::nn::hid::DebugPadButton::Left>(true);
    autoPilotState.buttons.Set<::nn::hid::DebugPadButton::Up>(true);
    autoPilotState.buttons.Set<::nn::hid::DebugPadButton::Right>(true);
    autoPilotState.buttons.Set<::nn::hid::DebugPadButton::Down>(true);

    // 変域を超過
    autoPilotState.analogStickR.x = ::nn::hid::AnalogStickMax;
    autoPilotState.analogStickR.y = ::nn::hid::AnalogStickMax;
    autoPilotState.analogStickL.x = -::nn::hid::AnalogStickMax;
    autoPilotState.analogStickL.y = -::nn::hid::AnalogStickMax;

    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    ::nn::hid::DebugPadState& state = g_States1[0];
    ::nn::hid::GetDebugPadState(&state);

    ::nnt::hid::FinalizeDebugPadForAutoPilot();

    // 未定義のフラグは無視されている
    EXPECT_EQ(1, state.attributes.CountPopulation());
    EXPECT_TRUE(
        state.attributes.Test<::nn::hid::DebugPadAttribute::IsConnected>());

    // 禁則処理が保証されている
    EXPECT_TRUE(state.buttons.Test<::nn::hid::DebugPadButton::Left>());
    EXPECT_TRUE(state.buttons.Test<::nn::hid::DebugPadButton::Up>());
    EXPECT_FALSE(state.buttons.Test<::nn::hid::DebugPadButton::Right>());
    EXPECT_FALSE(state.buttons.Test<::nn::hid::DebugPadButton::Down>());

    const ::nn::hid::AnalogStickState analogSticks[] = {
        state.analogStickR,
        state.analogStickL
    };

    for (int i = 0; i < 2; ++i)
    {
        SCOPED_TRACE(i == 0 ? "state.analogStickR" : "state.analogStickL");
        // 変域が保証されている
        const int64_t x = analogSticks[i].x;
        const int64_t y = analogSticks[i].y;
        EXPECT_LE(static_cast<int32_t>(::std::sqrt(x * x + y * y)),
                  ::nn::hid::AnalogStickMax);
    }
}

//!< DebugPad の入力状態は自動操作状態の設定を正しく反映するか
TEST(DebugPadAutoPilotSuite, StateReadingTest1)
{
    ::nn::hid::InitializeDebugPad();

    ::nn::hid::GetDebugPadState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultDebugPadState(g_States1[0]);
    }

    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::DebugPadStateCountMax / 3; ++i)
    {
        FuzzDebugPadAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetDebugPadSamplingInterval()
                        .GetMilliSeconds()));
    }

    int count = ::nn::hid::GetDebugPadStates(
        g_States1, ::nn::hid::DebugPadStateCountMax);
    EXPECT_LE(1, count);

    bool isActive = true;

    // DebugPadDriver の状態に応じて DebugPad の入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::DebugPadState& lhs = g_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (lhs.buttons.IsAllOff())
        {
            isActive = false;
        }
        else
        {
            SCOPED_TRACE("");
            EXPECT_TRUE(isActive);
            ExpectFuzzedDebugPadState(lhs);
        }

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentDebugPadStates(lhs, g_States1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeDebugPadForAutoPilot();
}

//!< DebugPad の入力状態は自動操作状態の解除を正しく反映するか
TEST(DebugPadAutoPilotSuite, StateReadingTest2)
{
    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState;
    FuzzDebugPadAutoPilotState(&autoPilotState, 10);

    ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetDebugPadAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::DebugPadStateCountMax / 2 *
            ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetDebugPadStates(g_States1,
                                             ::nn::hid::DebugPadStateCountMax);
    EXPECT_EQ(::nn::hid::DebugPadStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::DebugPadState state = {};

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            const ::nn::hid::DebugPadState& lhs = g_States1[i];
            const ::nn::hid::DebugPadState& rhs = g_States1[i + 1];
            SCOPED_TRACE("");
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        if (g_States1[i].analogStickR.x != state.analogStickR.x)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.attributes = autoPilotState.attributes;
            state.buttons = autoPilotState.buttons;
            state.analogStickR = autoPilotState.analogStickR;
            state.analogStickL = autoPilotState.analogStickL;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginDebugPadStates(state, g_States1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeDebugPadForAutoPilot();
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST(DebugPadAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::InitializeDebugPad();

    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::DebugPadStateCountMax * 3; ++i)
    {
        FuzzDebugPadAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetDebugPadAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetDebugPadSamplingInterval().GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::DebugPadState *ptr1 = (i % 2 == 1) ? &g_States1[0]
                                                      : &g_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::DebugPadState *ptr2 = (i % 2 == 1) ? &g_States2[0]
                                                      : &g_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count =
            ::nn::hid::GetDebugPadStates(ptr1,
                                         ::nn::hid::DebugPadStateCountMax);
        EXPECT_LE(lastCount, count);

        bool isActive = true;

        for (int j = 0; j < count; ++j)
        {
            if (ptr1[j].buttons.IsAllOff())
            {
                isActive = false;
                EXPECT_LE(0, ptr1[i].samplingNumber);
            }
            else
            {
                SCOPED_TRACE("");
                EXPECT_TRUE(isActive);
                ExpectFuzzedDebugPadState(ptr1[j]);
            }

            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentDebugPadStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginDebugPadStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::DebugPadStateCountMax, lastCount);

    ::nnt::hid::FinalizeDebugPadForAutoPilot();
}

namespace
{

void FuzzDebugPadAutoPilotState(
    ::nn::hid::debug::DebugPadAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    pState->attributes.Reset();
    pState->attributes.Set<::nn::hid::DebugPadAttribute::IsConnected>();
    pState->buttons.Reset();
    pState->buttons.Set(
        seed % (::nn::hid::DebugPadButton::Down::Index + 1), true);
    pState->analogStickR.x = seed * 3;
    pState->analogStickR.y = seed * 5;
    pState->analogStickL.x = seed * 7;
    pState->analogStickL.y = seed * 11;
}

void ExpectDefaultDebugPadState(
    const ::nn::hid::DebugPadState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_FALSE(
        state.attributes.Test<::nn::hid::DebugPadAttribute::IsConnected>());
    EXPECT_TRUE(state.buttons.IsAllOff());
    EXPECT_EQ(0, state.analogStickR.x);
    EXPECT_EQ(0, state.analogStickR.y);
    EXPECT_EQ(0, state.analogStickL.x);
    EXPECT_EQ(0, state.analogStickL.y);
}

void ExpectFuzzedDebugPadState(
    const ::nn::hid::DebugPadState& state) NN_NOEXCEPT
{
    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(0, state.analogStickR.x % 3);
    EXPECT_EQ(0, state.analogStickR.y % 5);
    EXPECT_EQ(0, state.analogStickL.x % 7);
    EXPECT_EQ(0, state.analogStickL.y % 11);
    const int seed = state.analogStickR.x / 3;
    ::nn::hid::DebugPadButtonSet buttons = {};
    buttons.Set(seed % (::nn::hid::DebugPadButton::Down::Index + 1), true);
    EXPECT_TRUE(
        state.attributes.Test<::nn::hid::DebugPadAttribute::IsConnected>());
    EXPECT_EQ(buttons, state.buttons);
    EXPECT_EQ(seed, state.analogStickR.x / 3);
    EXPECT_EQ(seed, state.analogStickR.y / 5);
    EXPECT_EQ(seed, state.analogStickL.x / 7);
    EXPECT_EQ(seed, state.analogStickL.y / 11);
}

void ExpectSameOriginDebugPadStates(
    const ::nn::hid::DebugPadState& lhs,
    const ::nn::hid::DebugPadState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.attributes, rhs.attributes);
    EXPECT_EQ(lhs.buttons, rhs.buttons);
    EXPECT_EQ(lhs.analogStickR.x, rhs.analogStickR.x);
    EXPECT_EQ(lhs.analogStickR.y, rhs.analogStickR.y);
    EXPECT_EQ(lhs.analogStickL.x, rhs.analogStickL.x);
    EXPECT_EQ(lhs.analogStickL.y, rhs.analogStickL.y);
}

void ExpectAdjacentDebugPadStates(
    const ::nn::hid::DebugPadState& lhs,
    const ::nn::hid::DebugPadState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
    EXPECT_GE(lhs.analogStickR.x, rhs.analogStickR.x);
    EXPECT_GE(lhs.analogStickR.y, rhs.analogStickR.y);
    EXPECT_GE(lhs.analogStickL.x, rhs.analogStickL.x);
    EXPECT_GE(lhs.analogStickL.y, rhs.analogStickL.y);
}

} // namespace
