﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_ConsoleSixAxisSensor.h"

#define VERBOSE

namespace
{

const int StateCountMax = ::nn::hid::ConsoleSixAxisSensorStateCountMax;
::nn::hid::tmp::SixAxisSensorCountState g_States[StateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(ConsoleSixAxisSensorReadingStateSuiteForDevelop, ReadingStateTest1)
{
    for (::nn::hid::tmp::SixAxisSensorCountState& state : g_States)
    {
        state = ::nn::hid::tmp::SixAxisSensorCountState();
        state.samplingNumber = 0xDEADBEEF;
    }

    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::StartSixAxisSensor(handle);

    // バッファの半分だけセンサー値が溜まるまで待つ
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            StateCountMax / 2 * ::nnt::hid::GetConsoleSixAxisSensorDriverSamplingInterval().GetMilliSeconds()
        )
    );

    int count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States,
                                                                   StateCountMax,
                                                                   handle);
    EXPECT_GE(StateCountMax, count);

    for (int i = count; i < StateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
    }

    ::nn::hid::StopSixAxisSensor(handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< 読みだした入力状態の値は適切か
TEST(ConsoleSixAxisSensorReadingStateSuiteForDevelop, ReadingStateTest2)
{
    for (::nn::hid::tmp::SixAxisSensorCountState& state : g_States)
    {
        state = ::nn::hid::tmp::SixAxisSensorCountState();
    }

    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    const int ReadTestCount = 5; // 読み出し試行回数

    for (int i = 0; i < ReadTestCount; i++)
    {
        int count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States, StateCountMax, handle);

        EXPECT_GE(StateCountMax, count);

#ifdef VERBOSE
        NN_LOG("[%lld] Acc : (%d, %d, %d), Gyro : (%d, %d, %d)\n",
            g_States[0].samplingNumber,
            g_States[0].acceleration.x,
            g_States[0].acceleration.y,
            g_States[0].acceleration.z,
            g_States[0].angularVelocity.x,
            g_States[0].angularVelocity.y,
            g_States[0].angularVelocity.z
        );
#endif // VERBOSE

        auto tempState = g_States[0];
        for (int i = 1; i < count; i++)
        {
            // サンプル番号の連続性
            EXPECT_EQ(tempState.samplingNumber - 1, g_States[i].samplingNumber);

            // tick の単調増加性
            EXPECT_GE(tempState.tick, g_States[i].tick);

            tempState = g_States[i];
        }

        // 1 周期待つ
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds()
            )
        );
    }

    ::nn::hid::StopSixAxisSensor(handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< 読みだした入力状態のサンプリング番号の増加量は適切か
TEST(ConsoleSixAxisSensorReadingStateSuiteForDevelop, ReadingStateTest3)
{
    for (::nn::hid::tmp::SixAxisSensorCountState& state : g_States)
    {
        state = ::nn::hid::tmp::SixAxisSensorCountState();
    }

    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    // 計測開始時点のサンプリング番号を取得
    auto count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States, StateCountMax, handle);

    EXPECT_GE(StateCountMax, count);

    const auto SamplingNumber0 = g_States[0].samplingNumber;

    // 所定時間待つ
    const auto ExpectedSamplingNumberDelta = 1000; //!< 期待されるサンプル数の増加分
    ::nnt::hid::SleepThreadForSampleCount(ExpectedSamplingNumberDelta);

    // 計測終了時点のサンプリング番号を取得
    count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States, StateCountMax, handle);

    EXPECT_GE(StateCountMax, count);

    const auto ActualSamplingNumberDelta = g_States[0].samplingNumber - SamplingNumber0;

    NN_LOG("ActualSamplingNumberDelta : %lld\n", ActualSamplingNumberDelta);

    EXPECT_NEAR(ActualSamplingNumberDelta,
                ExpectedSamplingNumberDelta,
                ExpectedSamplingNumberDelta * 0.05f);

    ::nn::hid::StopSixAxisSensor(handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< キャリブレーション値を読み出せるか
TEST(ConsoleSixAxisSensorReadingCalibrationValuesSuiteForDevelop, CalibrationValueTest1)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::tmp::ConsoleSixAxisSensorCalibrationValues values = ::nn::hid::tmp::ConsoleSixAxisSensorCalibrationValues();

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(100)
    );

    const auto TempValues = values;
    ::nn::hid::tmp::GetConsoleSixAxisSensorCalibrationValues(&values, handle);

    // 値が変わっているか確認
    EXPECT_NE(0, ::std::memcmp(&values, &TempValues, sizeof(::nn::hid::tmp::ConsoleSixAxisSensorCalibrationValues)));

    NN_LOG("[AccZeroCount] (%d, %d, %d)\n",
        values.accelerationZeroCount.x,
        values.accelerationZeroCount.y,
        values.accelerationZeroCount.z);
    NN_LOG("[AccSensitivity] (%d, %d, %d)\n",
        values.accelerationSensitivityCount.x,
        values.accelerationSensitivityCount.y,
        values.accelerationSensitivityCount.z);
    NN_LOG("[GyroZeroCount] (%d, %d, %d)\n",
        values.angularVelocityZeroCount.x,
        values.angularVelocityZeroCount.y,
        values.angularVelocityZeroCount.z);
    NN_LOG("[GyroSensitivity] (%d, %d, %d)\n",
        values.angularVelocitySensitivityCount.x,
        values.angularVelocitySensitivityCount.y,
        values.angularVelocitySensitivityCount.z);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(ConsoleSixAxisSensorReadingStateSuiteForDevelop, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(ConsoleSixAxisSensorReadingStateSuiteForDevelop, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::StartSixAxisSensor(handle);

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States, StateCountMax, handle);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::tmp::SixAxisSensorCountState& lhs = g_States[j];
            const ::nn::hid::tmp::SixAxisSensorCountState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(StateCountMax, lastCount);

    ::nn::hid::StopSixAxisSensor(handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
