﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_CaptureButton.h"

namespace
{
::nn::hid::system::CaptureButtonState g_State;
} // namespace

//!< 撮影ボタンの初期化状態は GetCaptureButtonStates() の呼び出しに反映されるか
TEST(CaptureButtonInitializationSuite, NoInitializationTest1)
{
    // 撮影ボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetCaptureButtonStates(&g_State, 1), "");

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンが終了しきっていないので成功する。
    ::nn::hid::system::GetCaptureButtonStates(&g_State, 1);

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetCaptureButtonStates(&g_State, 1), "");
}

//!< 撮影ボタンの初期化状態は SetCaptureButtonAutoPilotState() の呼び出しに反映されるか
TEST(CaptureButtonInitializationSuite, NoInitializationTest2)
{
    ::nn::hid::debug::CaptureButtonAutoPilotState autoPilotState = {};

    // 撮影ボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState), "");

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンが終了しきっていないので成功する。
    ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState);

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState), "");
}

//!< 撮影ボタンの初期化状態は UnsetCaptureButtonAutoPilotState() の呼び出しに反映されるか
TEST(CaptureButtonInitializationSuite, NoInitializationTest3)
{
    // 撮影ボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetCaptureButtonAutoPilotState(), "");

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンが終了しきっていないので成功する。
    ::nn::hid::debug::UnsetCaptureButtonAutoPilotState();

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetCaptureButtonAutoPilotState(), "");
}

//!< 撮影ボタンの初期化処理は最低一つの入力状態の取得を保証するか
TEST(CaptureButtonInitializationSuite, StateReadingTest1)
{
    ::nn::hid::system::InitializeCaptureButton();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::system::GetCaptureButtonStates(&g_State, 1);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_TRUE(g_State.buttons.IsAllOff());

    ::nn::hid::debug::FinalizeCaptureButton();
}

//!< 撮影ボタンの初期化処理はサンプリングを正しく有効化するか
TEST(CaptureButtonInitializationSuite, StateReadingTest2)
{
    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::GetCaptureButtonStates(&g_State, 1);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeCaptureButton();

    // 撮影ボタンが終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::CaptureButtonStateCountMax *
            ::nnt::hid::GetCaptureButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::GetCaptureButtonStates(&g_State, 1);

    EXPECT_LT(
        samplingNumber + ::nn::hid::system::CaptureButtonStateCountMax / 4,
        g_State.samplingNumber);

    ::nn::hid::debug::FinalizeCaptureButton();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // 撮影ボタンが終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::CaptureButtonStateCountMax *
            ::nnt::hid::GetCaptureButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::system::GetCaptureButtonStates(&g_State, 1);

    EXPECT_GT(
        samplingNumber + ::nn::hid::system::CaptureButtonStateCountMax / 4,
        g_State.samplingNumber);

    ::nn::hid::debug::FinalizeCaptureButton();
#endif
}
