﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nnt/nntest.h>
#include <algorithm>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
    #include <nv/nv_ServiceName.h>
#endif

#include "HdcpRawTestInitializer.h"
#include "testHdcp_Config.h"

#define GFX_INIT_FINAL_WAR
/////////////////////////////////////////////////////////////////////////////////////////
// FIX:
//
// Currently, nv::FinalizeGraphics cannot finalize nv::Graphics completely.
// If you re-initialize nv::Graphics after nv::FinalizeGraphics,
//  the following messages will appear:
//
// Case 1:
//    Abort: 'NN_ABORT' in SetGraphicsAllocator() at pid = 127, tid = 510(MainThread)
//    C :/dvs/git/dirty/git - master_hos/core - hos/utils/nvos/hos/nvos_hos.cpp(2317)
//    nv::SetGraphicsAllocator must not be called more than once.
//
// If you re-initialize nv::Graphics without SetGraphicsAllocator,
//  the following messages will appear as well:
//
// Case 2:
//    Abort: 'NN_ABORT' in SetGraphicsServiceName() at pid = 128, tid = 520(MainThread)
//    C :/dvs/git/dirty/git - master_hos/core - hos/utils/nvos/hos/nvos_hos.cpp(2343)
//    nv::SetGraphicsServiceName must be called before nv::InitializeGraphics.
//
/////////////////////////////////////////////////////////////////////////////////////////

namespace nnt { namespace hdcp {

namespace {

const size_t GraphicsSystemMemorySize = 512 * 1024;

// Not thread safe...
int g_GraphicsInitializedCount = 0;

nn::os::ThreadType g_Thread;
const int ThreadStackSize = 16 * 1024;
NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
volatile bool g_IsThreadRunning;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// Use of nv::InitializeGraphics is required by using nvhdcp.
//------------------------------------------------------------------------------
void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

void ImposeStressOnDpcd(void* arg)
{
    auto pVideo = reinterpret_cast<nns::VideoConfigUtility*>(arg);
    int resolution480p = pVideo->GetConfigMode(720, 480);
    int resolution720p = pVideo->GetConfigMode(1280, 720);
    NN_LOG("resolution480p:[%d] resolution720p:[%d]\n", resolution480p, resolution720p);
    pVideo->SetDisplayPowerMode(HWC_POWER_MODE_OFF);
    pVideo->SetDisplayPowerMode(HWC_POWER_MODE_NORMAL);
    for (int i1=0; g_IsThreadRunning; ++i1)
    {
        NN_LOG(" - # of switching resolution:[%d]\n", i1);
        pVideo->SetActiveConfigMode(resolution720p);
        pVideo->SetActiveConfigMode(resolution480p);
    }
    NN_LOG(" - Consecutive resolution switch is done\n");
}

}

NN_IMPLICIT HdcpRawTestInitializer::HdcpRawTestInitializer() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    if (0 == g_GraphicsInitializedCount)
    {
        // グラフィックスシステムのためのメモリ周りの初期化を行います。
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsServiceName("nvdrv:t");
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
    ++g_GraphicsInitializedCount;
#endif
    m_pVideo = new nns::VideoConfigUtility(android::ISurfaceComposer::eDisplayIdHdmi);
}

HdcpRawTestInitializer::~HdcpRawTestInitializer() NN_NOEXCEPT
{
    delete m_pVideo;
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#ifndef GFX_INIT_FINAL_WAR
    if (1 == g_GraphicsInitializedCount)
    {
        // グラフィックスシステムのためのメモリ周りの初期化を行います。
        nv::FinalizeGraphics();
    }
    --g_GraphicsInitializedCount;
#endif
#endif
}

bool HdcpRawTestInitializer::IsHdmiConnected() NN_NOEXCEPT
{
    return m_pVideo->IsDisplayConnected();
}

void HdcpRawTestInitializer::ShowWarningBasedOnHdmiConnection(bool isConnectionExcected) NN_NOEXCEPT
{
    bool isConnected = m_pVideo->IsDisplayConnected();
    if (!isConnectionExcected && isConnected)
    {
        NN_LOG("--------------------------------------------------------------------------\n");
        NN_LOG(" Warning - This unit test supports a case just when HDMI is disconnected. \n");
        NN_LOG("--------------------------------------------------------------------------\n");
        m_pVideo->ShowDisplayInfo();
    }
    else if (isConnectionExcected && !isConnected)
    {
        NN_LOG("--------------------------------------------------------------------------\n");
        NN_LOG(" Warning - This unit test supports a case just when HDMI is connected. \n");
        NN_LOG("--------------------------------------------------------------------------\n");
    }
}

bool HdcpRawTestInitializer::StartThreadToImposeStressOnDpcd() NN_NOEXCEPT
{
    nn::Result result = nn::os::CreateThread(
        &g_Thread, ImposeStressOnDpcd, m_pVideo,
        g_ThreadStack, ThreadStackSize, nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    g_IsThreadRunning = result.IsSuccess();
    if (g_IsThreadRunning)
    {
        nn::os::StartThread(&g_Thread);
        return true;
    }
    NN_LOG("--------------------------------------------------------------------------\n");
    NN_LOG(" Warning - Cannot create thread to impose stress on DPCD. \n");
    NN_LOG("--------------------------------------------------------------------------\n");
    return false;
}

void HdcpRawTestInitializer::StopThreadToImposeStressOnDpcd() NN_NOEXCEPT
{
    if (g_IsThreadRunning)
    {
        g_IsThreadRunning = false;
        nn::os::WaitThread(&g_Thread);
    }
}

void HdcpRawTestInitializer::ChangeDisplayResolution() NN_NOEXCEPT
{
    int resolution480p = m_pVideo->GetConfigMode(720, 480);
    int resolution720p = m_pVideo->GetConfigMode(1280, 720);
    int activeConfig = m_pVideo->GetActiveConfigMode();
    m_pVideo->SetActiveConfigMode(activeConfig == resolution480p ? resolution720p : resolution480p);
}

}}  // namespace nnt::hdcp
